<?php

namespace Drupal\acsf_module_listing\Services;

use Drupal\acsf_module_listing\Entity\AcsfEnvironmentEntity;
use Drupal\Component\Utility\Html;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Provides SSH Drush service.
 */
class SshShellService {

  /**
   * Config service.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   *   Config service variable.
   */
  protected ImmutableConfig $config;

  /**
   * Cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   *   Cache service variable.
   */
  protected CacheBackendInterface $cache;

  /**
   * Logger channel service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   *   Logger channel service variable.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   *   Messenger service variable.
   */
  protected MessengerInterface $messenger;

  /**
   * SshShellService constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   Cache service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannelFactory
   *   Logger channel factory service.
   */
  public function __construct(ConfigFactoryInterface $configFactory, CacheBackendInterface $cache, LoggerChannelFactoryInterface $loggerChannelFactory, MessengerInterface $messenger) {
    $this->config = $configFactory->get('acsf_module_listing.modules_usage_configuration');
    $this->cache = $cache;
    $this->logger = $loggerChannelFactory->get('acsf_module_listing');
    $this->messenger = $messenger;
  }

  /**
   * Establish SSH Connection function.
   *
   * @param \Drupal\acsf_module_listing\Entity\AcsfEnvironmentEntity $acsf_environment
   *   ACSF environment entity variable.
   *
   * @return resource
   *   Connection resource.
   */
  public function establishSshConnection(AcsfEnvironmentEntity $acsf_environment) {
    try {
      $connection = ssh2_connect($acsf_environment->getVariable('ssh_url'), 22, ['hostkey' => 'ssh-rsa']);
      if ($connection) {
        $passphrase = $acsf_environment->getVariable('pass_phrase') ?: '';
        $auth = ssh2_auth_pubkey_file(
          $connection,
          $acsf_environment->getVariable('ssh_user'),
          $acsf_environment->getVariable('public_key_path'),
          $acsf_environment->getVariable('private_key_path'),
          $passphrase,
        );
        if ($auth) {
          return $connection;
        }
        $this->logger->error('Cannot authenticate using public/private keys pair. Please, verify your configuration.');
        return NULL;
      }
      $this->logger->error('SSH @ssh server seems unavailable.', ['@ssh' => $acsf_environment->getVariable('ssh_url')]);
      return NULL;

    } catch (\Exception $exception) {
      $this->logger->error('Cannot establish SSH connection due to the "%message"', ['%message' => $exception->getMessage()]);
    }
    $this->messenger->addError('Something went wrong. Please check database logs.');

    return NULL;
  }

  /**
   * Return modules list by website URL.
   *
   * @param \Drupal\acsf_module_listing\Entity\AcsfEnvironmentEntity $acsf_environment
   *   ACSF Environment entity.
   * @param string $website
   *   Website URL.
   * @param string $input
   *   Search string.
   * @param bool $ignore_cache
   *   Ignore cache storage or not.
   * @param bool $full_list
   *   False if only enabled modules.
   *
   * @return array
   *   Module array data.
   */
  public function getWebsiteModulesList(AcsfEnvironmentEntity $acsf_environment, string $website, string $input, bool $ignore_cache, bool $full_list = FALSE): array {
    $time = time() + $this->config->get('configuration.cache_lifetime');
    $input = explode(" ", $input);
    $key = 'acsf_module_listing.sites.' . $acsf_environment->id() . '.' . Html::cleanCssIdentifier($website);

    $cache = $this->cache->get($key);
    $modules_usage_response = NULL;

    if (isset($cache->data) && !$ignore_cache) {
      $modules_usage_response = $cache->data;
    }
    else {
      $ssh_connection = $this->establishSshConnection($acsf_environment);
      if ($ssh_connection) {
        $modules_usage_response = $this->getSshModulesList($ssh_connection, $acsf_environment, $website, $full_list);

        if ($modules_usage_response) {
          $this->cache->set($key, $modules_usage_response, $time);
        }
      }
    }

    $result = [];
    if ($modules_usage_response) {
      $modules = explode(PHP_EOL, $modules_usage_response);
      foreach ($modules as $key => $module_data) {
        $module_data = preg_split('/\s+/', $module_data);
        $modules[$key] = $module_data;
        if ($full_list) {
          $result[] = $module_data;
          continue;
        }

        $match = FALSE;
        foreach ($module_data as $module_item) {
          foreach ($input as $word) {
            if (!$match && str_contains(strtolower($module_item), $word)) {
              $match = TRUE;
              $result[] = $module_data;
            }
          }
        }
      }
    }

    return $result;
  }

  /**
   * Builder function that prepare drush command with modules list.
   *
   * @param resource $ssh_connection
   *   SSH Connection resource variable.
   * @param \Drupal\acsf_module_listing\Entity\AcsfEnvironmentEntity $acsf_environment
   *   ACSF Environment Entity variable.
   * @param string $website
   *   Website url as a location for pm-list command execution.
   * @param bool $full_list
   *   Return data for enabled modules or full list including disabled and
   *   themes.
   *
   * @return null|string
   *   Returns ssh output NULL in case of wrong connection attempt.
   */
  private function getSshModulesList($ssh_connection, AcsfEnvironmentEntity $acsf_environment, string $website, bool $full_list): null|string {
    $cmd = "drush -r {$acsf_environment->getVariable('drush_path')} -l {$website} pm-list ";
    // Add options.
    if (!$full_list) {
      $cmd .= "--type=Module --status=enabled";
    }

    try {
      $stream = ssh2_exec($ssh_connection, $cmd);

      if ($stream) {
        stream_set_blocking($stream, TRUE);

        $output = stream_get_contents($stream);
        fclose($stream);
        return $output;
      }
    } catch (\Exception $exception) {
      $this->logger->error($exception->getMessage(), ['exception' => $exception]);
    }
    $this->messenger->addError('Something went wrong. Please check database logs.');

    return NULL;
  }

}
