<?php

namespace Drupal\action_link\Plugin\ActionLinkStyle;

use Drupal\action_link\Entity\ActionLinkInterface;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Component\Utility\Html;
use Drupal\Core\Link;
use Drupal\Core\Session\AccountInterface;

/**
 * Base class for Action Link Style plugins.
 */
abstract class ActionLinkStyleBase extends PluginBase implements ActionLinkStyleInterface {

  /**
   * {@inheritdoc}
   */
  public function alterLinkBuild(array &$build, ActionLinkInterface $action_link, AccountInterface $user, array $dynamic_parameters, array $scalar_parameters): void {
    if ($build['#url']) {
      $link = Link::fromTextAndUrl($build['#label'], $build['#url']);

      $build['link'] = $link->toRenderable();

      // Set nofollow to prevent search bots from crawling anonymous links.
      $build['link']['#attributes']['rel'][] = 'nofollow';
    }
  }

  /**
   * {@inheritdoc}
   */
  public function alterLinksBuild(array &$build, ActionLinkInterface $action_link, AccountInterface $user, array $dynamic_parameters, array $scalar_parameters) {
    // Do nothing by default.
  }

  /**
   * {@inheritdoc}
   */
  public function getCsrfTokenHttpMethod(): string|false {
    return $this->pluginDefinition['csrf_token_http_method'];
  }

  /**
   * Creates a unique HTML class for an action link.
   *
   * We don't use \Drupal\Component\Utility\Html\HTML::getUniqueId() because we
   * want the same class to be used on all instances of the same action link, so
   * that they are all replaced.
   *
   * @param \Drupal\action_link\Entity\ActionLinkInterface $action_link
   *   The action link entity.
   * @param string $direction
   *   The direction.
   * @param \Drupal\Core\Session\AccountInterface $user
   *   The user account the action is for.
   * @param mixed ...$scalar_parameters
   *   The dynamic parameters for the action.
   *
   * @return string
   *   A CSS class.
   */
  protected function createCssIdentifier(ActionLinkInterface $action_link, string $direction, AccountInterface $user, ...$scalar_parameters): string {
    return Html::cleanCssIdentifier(implode(
      '-', [
        'action-link',
        $action_link->id(),
        $direction,
        $user->id(),
        ...array_values($scalar_parameters),
      ]
    ));
  }

}
