<?php

namespace Drupal\action_link\Plugin\ActionLinkStyle;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\action_link\Attribute\ActionLinkStyle;
use Drupal\action_link\Entity\ActionLinkInterface;
use Drupal\action_link\Plugin\ActionLinkStyleAjaxResponseTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Link style which uses a JavaScript link without reloading the page.
 *
 * The action link message is shown briefly alongside the clicked link.
 *
 * This gracefully degrades to the Nojs link style if JavaScript is not
 * available.
 *
 * @see templates/action-link-popup-message.html.twig
 */
#[ActionLinkStyle(
  id: 'ajax',
  label: new TranslatableMarkup('JavaScript'),
  description: new TranslatableMarkup('A link which makes an AJAX JavaScript GET request without reloading the page.'),
  csrf_token_http_method: Request::METHOD_GET,
)]
class Ajax extends ActionLinkStyleBase implements ContainerFactoryPluginInterface {

  use ActionLinkStyleAjaxResponseTrait;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('renderer'),
    );
  }

  /**
   * Creates an Ajax instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    RendererInterface $renderer
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public function alterLinkBuild(array &$build, ActionLinkInterface $action_link, AccountInterface $user, array $dynamic_parameters, array $scalar_parameters): void {
    parent::alterLinkBuild($build, $action_link, $user, $dynamic_parameters, $scalar_parameters);

    if ($build['link']) {
      // Graceful degradation: change the style plugin in the link URL to 'nojs'
      // so that if JavaScript is not enabled, the reload page plugin is used
      // instead. Drupal's AJAX system will replace this path component with
      // 'ajax' if JavaScript is enabled.
      // @todo This won't work because the link URL being changed by JavaScript
      // means that the CSRF token is invalid. Uncomment this when
      // https://www.drupal.org/project/drupal/issues/2670798 is fixed.
      // $route_parameters = $build['link']['#url']->getRouteParameters();
      // $route_parameters['link_style'] = 'nojs';
      // $build['link']['#url']->setRouteParameters($route_parameters);

      // Add the 'use-ajax' class to the link. This makes core handle the link
      // using a JS request and degrades gracefully to be handled by the nojs
      // link style plugin.
      $build['link']['#attributes']['class'][] = 'use-ajax';

      $build['#attached']['library'][] = 'action_link/link_style.ajax';
    }

    // Add a unique class to the outer HTML for the AJAX replacement. This is
    // added even if the link is not active, so that it can be replaced as part
    // of the response to another link.
    $build['#attributes']['class'][] = $this->createCssIdentifier($action_link, $build['#direction'], $user, ...$scalar_parameters);
  }

}
