<?php

declare(strict_types=1);

namespace Drupal\action_link\Plugin\ActionLinkStyle;

use Drupal\action_link\Attribute\ActionLinkStyle;
use Drupal\action_link\Entity\ActionLinkInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Link style which reloads the page with a button styled to look like a link.
 */
#[ActionLinkStyle(
  id: 'post_link',
  label: new TranslatableMarkup('Reload link using POST'),
  description: new TranslatableMarkup('A button styled to look like a link. This is more secure than a GET link.'),
  csrf_token_http_method: Request::METHOD_POST,
)]
class PostLink extends ActionLinkStyleBase implements ContainerFactoryPluginInterface {

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('messenger'),
    );
  }

  /**
   * Creates a PostLink instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    MessengerInterface $messenger
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public function alterLinkBuild(array &$build, ActionLinkInterface $action_link, AccountInterface $user, array $dynamic_parameters, array $scalar_parameters): void {
    if ($build['#url']) {
      $build['post_link'] = [
        '#theme' => 'post_link',
        '#url' => $build['#url'],
        '#label' => $build['#label'],
        '#token' => $build['#token'],
        // Clone the attributes so changes to them don't bleed to the parent
        // element.
        '#attributes' => clone $build['#attributes'],
        '#button_attributes' => clone $build['#attributes'],
        '#attached' => [
          'library' => ['action_link/post_link'],
        ],
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function handleActionRequest(bool $success, Request $request, RouteMatchInterface $route_match, ActionLinkInterface $action_link, string $direction, string $state, UserInterface $user, ...$parameters): Response {
    if ($success) {
      $message = $action_link->getMessage($direction, $state, ...$parameters);
    }
    else {
      $message = $action_link->getFailureMessage($direction, $state, ...$parameters);
    }

    if ($message) {
      $this->messenger->addMessage($message);
    }

    // Redirect to the referrer.
    if ($request->headers->get('referer')) {
      $response = new RedirectResponse($request->headers->get('referer'));
    }
    else {
      // This shouldn't happen outside of tests.
      $response = new RedirectResponse(Url::fromRoute('<front>')->toString());
    }

    return $response;
  }

}
