<?php

declare(strict_types=1);

namespace Drupal\action_link\Plugin\ActionLinkStyle;

use Drupal\action_link\Attribute\ActionLinkStyle;
use Drupal\action_link\Entity\ActionLinkInterface;
use Drupal\action_link\Plugin\ActionLinkStyleAjaxResponseTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Link style which makes JavaScript POST request without reloading the page.
 */
#[ActionLinkStyle(
  id: 'post_link_ajax',
  label: new TranslatableMarkup('JavaScript link using POST'),
  description: new TranslatableMarkup('A button styled to look like a link which makes an AJAX JavaScript POST request. This is more secure than a GET link.'),
  csrf_token_http_method: Request::METHOD_POST,
)]
class PostLinkAjax extends ActionLinkStyleBase implements ContainerFactoryPluginInterface {

  use ActionLinkStyleAjaxResponseTrait;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('renderer'),
    );
  }

  /**
   * Creates an Ajax instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    RendererInterface $renderer
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public function alterLinkBuild(array &$build, ActionLinkInterface $action_link, AccountInterface $user, array $dynamic_parameters, array $scalar_parameters): void {
    if ($build['#url']) {
      $build['post_link'] = [
        '#theme' => 'post_link',
        '#url' => $build['#url'],
        '#label' => $build['#label'],
        '#token' => $build['#token'],
        '#attributes' => clone $build['#attributes'],
        '#button_attributes' => clone $build['#attributes'],
        '#attached' => [
          'library' => [
            'action_link/post_link',
            'core/drupal.ajax',
          ],
        ],
      ];

      $build['#attached']['library'][] = 'action_link/link_style.ajax';

      $build['post_link']['#button_attributes']['class'][] = 'use-ajax-submit';

      // We need to add this library for AJAX button submit to work, even though
      // the library is marked as internal, because of a core bug.
      // @see https://www.drupal.org/project/drupal/issues/2755791.
      $build['#attached']['library'][] = 'core/internal.jquery.form';
    }

    // Add a unique class to the outer HTML for the AJAX replacement. This is
    // added even if the link is not active, so that it can be replaced as part
    // of the response to another link.
    $build['#attributes']['class'][] = $this->createCssIdentifier($action_link, $build['#direction'], $user, ...$scalar_parameters);
  }

}
