<?php

namespace Drupal\activetickets_client\Client;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;

/**
 * ActiveTickets client.
 */
class ActiveTicketsClient implements ActiveTicketsClientInterface {

  /**
   * Date format.
   *
   * @var string
   */
  const DATEFORMAT = 'Y-m-d\TH:i:s';

  /**
   * Soap client.
   *
   * @var \SoapClient
   */
  public \SoapClient $soapClient;

  /**
   * Logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  private LoggerChannelInterface $logger;

  /**
   * Config definition.
   *
   * @var \Drupal\Core\Config\Config
   */
  public Config $config;

  /**
   * Cache.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface|bool
   */
  protected CacheBackendInterface|bool $cache;

  /**
   * ActiveTicketsSoapClient constructor.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannel
   *   Logger definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface|bool $cache
   *   Cache backend.
   * @param bool $memberClient
   *   Is this a restricted instance of the client?
   *
   * @throws \Exception
   */
  public function __construct(LoggerChannelFactoryInterface $loggerChannel, ConfigFactoryInterface $configFactory, CacheBackendInterface|bool $cache, bool $memberClient = FALSE) {
    $this->logger = $loggerChannel->get('activetickets_client');
    $this->config = $configFactory->get('activetickets_client.settings');
    $this->cache = $cache;
    $this->initSoapClient($memberClient);
  }

  /**
   * Create new client.
   *
   * @return \Drupal\activetickets_client\Client\ActiveTicketsClient
   *   The ActiveTicketsClient object.
   *
   * @throws \Exception
   */
  public static function create(): ActiveTicketsClient {
    return new static(
      \Drupal::service('logger.factory')->get('activetickets_client'),
      \Drupal::service('config.factory'),
      \Drupal::cache(),
      FALSE
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig() {
    return $this->config;
  }

  /**
   * {@inheritdoc}
   */
  public function initSoapClient(bool $memberClient = FALSE) {
    $config_wsdl = $this->config->get('wsdl_url');
    if ($memberClient) {
      $config_wsdl = $this->config->get('wsdl_member_url');
    }

    try {
      $this->soapClient = new \SoapClient($config_wsdl);
    }
    catch (\SoapFault $soapFault) {
      $this->logger->error($soapFault->getMessage());
      throw new \Exception('Error during initialization SOAP object.');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getRequestParams(): array {
    return [
      'Clientname' => $this->config->get('client_name'),
      'LanguageCode' => $this->config->get('langcode'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getMemberRequestParams(): array {
    return [
      'ClientName' => $this->config->get('client_name'),
      'LanguageCode' => $this->config->get('langcode'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function parseXmlResult(string $xml_string): array {
    $return = [];
    $nodes = [];
    libxml_use_internal_errors(TRUE);
    $xml = simplexml_load_string($xml_string);

    if (!$xml) {
      $return['error'] = $xml_string;
      return $return;
    }

    foreach ($xml as $xmlNode) {
      $json = json_encode($xmlNode);
      $nodes[] = json_decode($json, TRUE);
    }

    $return['result'] = $nodes;

    return $return;
  }

  /**
   * {@inheritdoc}
   */
  public function get(array $params, string $method): array {
    $resultMethod = $method . 'Result';

    try {
      $response = $this->soapClient->{$method}($params);
      return $this->parseXmlResult($response->{$resultMethod});
    }
    catch (\SoapFault $soapFault) {
      $this->logger->error($soapFault->getMessage());
    }

    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function set(array $params, string $method): array {
    $resultMethod = $method . 'Result';
    $return = [];

    try {
      libxml_use_internal_errors(TRUE);
      $response = $this->soapClient->{$method}($params);
      $result = $response->{$resultMethod};

      if ($result === 'True') {
        $return['result'] = TRUE;
      }
      else {
        $xml = simplexml_load_string($result);

        if (!$xml) {
          $return['error'] = $result;
        }

        $return['result'] = $xml;
      }
    }
    catch (\SoapFault $soapFault) {
      $return['error'] = $soapFault->getMessage();
      $this->logger->error($soapFault->getMessage());
    }

    return $return;
  }

  /**
   * {@inheritDoc}
   */
  public function getCachedProgramList(\DateTime $from, \DateTime $to, int $genreId): array {
    $cache_id = 'activetickets_client:programlist:' . $genreId;

    if ($cache = $this->cache->get($cache_id)) {
      $programList = $cache->data;
    }
    else {
      $programList = $this->getProgramList($from, $to, $genreId);
      $this->cache->set($cache_id, $programList);
    }

    return $programList;
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramList(\DateTime $from, \DateTime $to, int $genreId = 0, string $csvKeywords = '', bool $includePrices = TRUE): array {
    $params = $this->getRequestParams();
    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
      'GenreId' => $genreId,
      'csvKeywords' => $csvKeywords,
      'IncludePrices' => $includePrices,
    ];

    return $this->get($params, 'GetProgramList');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramDeletedAndinActives(\DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();
    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
    ];

    return $this->get($params, 'GetProgramDeletedAndinActives');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramListWithCustomKey(\DateTime $from, \DateTime $to, int $genreId, string $csvKeywords, bool $includePrices): array {
    $params = $this->getRequestParams();
    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
      'GenreId' => $genreId,
      'csvKeywords' => $csvKeywords,
      'IncludePrices' => $includePrices,
    ];

    return $this->get($params, 'GetProgramListWithCustomKey');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramListWithCustomKeyIffr(\DateTime $from, \DateTime $to, int $genreId, string $csvKeywords, bool $includePrices) : array {
    $params = $this->getRequestParams();
    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
      'GenreId' => $genreId,
      'csvKeywords' => $csvKeywords,
      'IncludePrices' => $includePrices,
    ];

    return $this->get($params, 'GetProgramListWithCustomKeyIFFR');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramListBig(\DateTime $from, \DateTime $to, int $genreId = 0, string $csvKeywords = '', bool $includePrices = TRUE): array {
    $params = $this->getRequestParams();
    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
      'GenreId' => $genreId,
      'csvKeywords' => $csvKeywords,
      'IncludePrices' => $includePrices,
    ];

    return $this->get($params, 'GetProgramListBig');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramListWithCapacities(\DateTime $from, \DateTime $to, int $genreId, string $csvKeywords, bool $includePrices, bool $includeCapacities) : array {
    $params = $this->getRequestParams();

    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
      'GenreId' => $genreId,
      'csvKeywords' => $csvKeywords,
      'IncludePrices' => $includePrices,
      'IncludeCapacities' => $includeCapacities,
    ];

    return $this->get($params, 'GetProgramListWithCapacities');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramListWithCapacitiesAndSold(\DateTime $from, \DateTime $to, int $genreId, string $csvKeywords, bool $includePrices, bool $includeCapacities) : array {
    $params = $this->getRequestParams();

    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
      'GenreId' => $genreId,
      'csvKeywords' => $csvKeywords,
      'IncludePrices' => $includePrices,
      'IncludeCapacities' => $includeCapacities,
    ];

    return $this->get($params, 'GetProgramListWithCapacitiesAndSold');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramCapacitiesByRang(): array {
    $params = $this->getRequestParams();

    return $this->get($params, 'GetProgramCapacitiesByRang');
  }

  /**
   * {@inheritdoc}
   */
  public function getGenreList(): array {
    $params = $this->getRequestParams();

    return $this->get($params, 'GetGenreList');
  }

  /**
   * {@inheritdoc}
   */
  public function getLocationList(): array {
    $params = $this->getRequestParams();

    return $this->get($params, 'GetLocationList');
  }

  /**
   * {@inheritdoc}
   */
  public function getCharacteristicsList(): array {
    $params = $this->getRequestParams();

    return $this->get($params, 'GetCharacteristicsList');
  }

  /**
   * {@inheritdoc}
   */
  public function getSubGenreList(): array {
    $params = $this->getRequestParams();

    return $this->get($params, 'GetSubGenreList');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramIdsViaSubgenresList(int $subGenreId, \DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();
    $params += [
      'subGenreId' => $subGenreId,
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
    ];

    return $this->get($params, 'GetProgramIdsViaSubgenresList');
  }

  /**
   * {@inheritdoc}
   */
  public function getSubgenresViaProgramList(int $programId): array {
    $params = $this->getRequestParams();
    $params += [
      'ProgramId' => $programId,
    ];

    return $this->get($params, 'GetSubgenresViaProgramList');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramDetail(int $programId): array {
    $params = $this->getRequestParams();
    $params += [
      'ProgramId' => $programId,
    ];

    return $this->get($params, 'GetProgramDetail');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramListLight(\DateTime $from, \DateTime $to, int $genreId, string $keywords): array {
    $params = $this->getRequestParams();

    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
      'GenreId' => $genreId,
      'Keywords' => $keywords,
    ];

    return $this->get($params, 'GetProgramListLight');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramConfirmed(\DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();

    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
    ];

    return $this->get($params, 'GetProgramConfirmed');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramStatus(int $programId): array {
    $params = $this->getRequestParams();
    $params += [
      'ProgramId' => $programId,
    ];

    return $this->get($params, 'GetProgramStatus');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramKijkwijzer(int $programId): array {
    $params = $this->getRequestParams();
    $params += [
      'ProgramId' => $programId,
    ];

    return $this->get($params, 'GetProgramKijkwijzer');
  }

  /**
   * {@inheritdoc}
   */
  public function getProgramPresale(int $programId): array {
    $params = $this->getRequestParams();
    $params += [
      'ProgramId' => $programId,
    ];

    return $this->get($params, 'GetProgramPresale');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitors(): array {
    $params = $this->getRequestParams();

    return $this->get($params, 'GetVisitors');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorsWithinModificationPeriod(\DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();
    $params += [
      'ModificationFrom' => $from->format(self::DATEFORMAT),
      'ModificationTo' => $to->format(self::DATEFORMAT),
    ];

    return $this->get($params, 'GetVisitorsWithinModificationPeriod');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorsWithinModificationPeriod2(\DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();
    $params += [
      'ModificationFrom' => $from->format(self::DATEFORMAT),
      'ModificationTo' => $to->format(self::DATEFORMAT),
    ];

    return $this->get($params, 'GetVisitorsWithinModificationPeriod2');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorsLight() : array {
    $params = $this->getRequestParams();
    return $this->get($params, 'GetVisitorsLight');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorsLightWithinCreationPeriod(\DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();
    $params += [
      'CreatedFrom' => $from->format(self::DATEFORMAT),
      'CreatedTo' => $to->format(self::DATEFORMAT),
    ];
    return $this->get($params, 'GetVisitorsLightWithinCreationPeriod');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorsLightWithinCreationPeriod2(\DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();
    $params += [
      'CreatedFrom' => $from->format(self::DATEFORMAT),
      'CreatedTo' => $to->format(self::DATEFORMAT),
    ];
    return $this->get($params, 'GetVisitorsLightWithinCreationPeriod2');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorsWithinCreationPeriod(\DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();
    $params += [
      'CreatedFrom' => $from->format(self::DATEFORMAT),
      'CreatedTo' => $to->format(self::DATEFORMAT),
    ];

    return $this->get($params, 'GetVisitorsWithinCreationPeriod');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorsWithinCreationPeriod2(\DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();
    $params += [
      'CreatedFrom' => $from->format(self::DATEFORMAT),
      'CreatedTo' => $to->format(self::DATEFORMAT),
    ];

    return $this->get($params, 'GetVisitorsWithinCreationPeriod2');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitor(int $visitorId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
    ];

    return $this->get($params, 'GetVisitor');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorBig(int $visitorId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
    ];

    return $this->get($params, 'GetVisitorBig');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorByCredentials(string $email, string $password): array {
    $params = $this->getRequestParams();
    $params += [
      'EmailAddress' => $email,
      'Password' => $password,
    ];

    return $this->get($params, 'GetVisitorByCredentials');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorByCredentials2(string $email, string $password): array {
    $params = $this->getRequestParams();
    $params += [
      'EmailAddress' => $email,
      'Password' => $password,
    ];

    return $this->get($params, 'GetVisitorByCredentials2');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorByEmail(string $email): array {
    $params = $this->getRequestParams();
    $params += [
      'EmailAddress' => $email,
    ];

    return $this->get($params, 'GetVisitorByEmail');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorByEmail2(string $email): array {
    $params = $this->getRequestParams();
    $params += [
      'EmailAddress' => $email,
    ];

    return $this->get($params, 'GetVisitorByEmail2');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorByEmailBig(string $email): array {
    $params = $this->getRequestParams();
    $params += [
      'EmailAddress' => $email,
    ];

    return $this->get($params, 'GetVisitorByEmailBig');
  }

  /**
   * {@inheritdoc}
   */
  public function setVisitorNew(string $email, string $password, string $initials, string $firstname, string $infix, string $lastname, string $street, string $housenumber, string $city, string $zipcode, int $countryId, string $phoneNumber1, string $phoneNumber2, \DateTime $dateOfBirth, string $gender, bool $onMailingList, string $bankAccount, string $organisation): array {
    $params = $this->getMemberRequestParams();
    $params += [
      'Email' => $email,
      'Password' => $password,
      'Initials' => $initials,
      'Firstname' => $firstname,
      'Infix' => $infix,
      'Lastname' => $lastname,
      'Street' => $street,
      'Housenumber' => $housenumber,
      'City' => $city,
      'Zipcode' => $zipcode,
      'CountryId' => $countryId,
      'Phonenumber1' => $phoneNumber1,
      'Phonenumber2' => $phoneNumber2,
      'DateOfBirth' => $dateOfBirth->format('Y-m-d'),
      'Gender' => $gender,
      'onMailingList' => $onMailingList,
      'bankAccount' => $bankAccount,
      'organisation' => $organisation,
    ];

    return $this->set($params, 'SetVisitorNew');
  }

  /**
   * {@inheritdoc}
   */
  public function setVisitorNewGender(string $email, string $password, string $initials, string $firstname, string $infix, string $lastname, string $street, string $housenumber, string $city, string $zipcode, int $countryId, string $phoneNumber1, string $phoneNumber2, \DateTime $dateOfBirth, string $gender, bool $onMailingList, string $bankAccount, string $organisation): array {
    $params = $this->getRequestParams();
    $params += [
      'Email' => $email,
      'Password' => $password,
      'Initials' => $initials,
      'Firstname' => $firstname,
      'Infix' => $infix,
      'Lastname' => $lastname,
      'Street' => $street,
      'Housenumber' => $housenumber,
      'City' => $city,
      'Zipcode' => $zipcode,
      'CountryId' => $countryId,
      'Phonenumber1' => $phoneNumber1,
      'Phonenumber2' => $phoneNumber2,
      'DateOfBirth' => $dateOfBirth->format('Y-m-d'),
      'Gender' => $gender,
      'onMailingList' => $onMailingList,
      'bankAccount' => $bankAccount,
      'organisation' => $organisation,
    ];

    return $this->set($params, 'SetVisitorNewGender');
  }

  /**
   * {@inheritdoc}
   */
  public function setVisitorDelete(int $visitorId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
    ];

    return $this->set($params, 'SetVisitorDelete');
  }

  /**
   * {@inheritdoc}
   */
  public function setVisitorUpdate(array $params): array {
    $requestParams = $this->getRequestParams();
    $requestParams += $params;
    return $this->set($requestParams, 'SetVisitorUpdate');
  }

  /**
   * {@inheritdoc}
   */
  public function setVisitorUpdateGender(int $visitorId, string $email, string $password, string $initials, string $firstname, string $infix, string $lastname, string $street, string $housenumber, string $city, string $zipcode, int $countryId, string $phoneNumber1, string $phoneNumber2, \DateTime $dateOfBirth, string $gender, bool $onMailingList, string $bankAccount, string $organisation): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
      'Email' => $email,
      'Password' => $password,
      'Initials' => $initials,
      'Firstname' => $firstname,
      'Infix' => $infix,
      'Lastname' => $lastname,
      'Street' => $street,
      'Housenumber' => $housenumber,
      'City' => $city,
      'Zipcode' => $zipcode,
      'CountryId' => $countryId,
      'Phonenumber1' => $phoneNumber1,
      'Phonenumber2' => $phoneNumber2,
      'DateOfBirth' => $dateOfBirth->format('Y-m-d'),
      'Gender' => $gender,
      'onMailingList' => $onMailingList,
      'bankAccount' => $bankAccount,
      'organisation' => $organisation,
    ];

    return $this->set($params, 'SetVisitorUpdateGender');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorWebserviceLoginKey(int $visitorId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
    ];

    return $this->set($params, 'GetVisitorWebserviceLoginKey');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorWebserviceLoginKeyByEmail(string $email): array {
    $params = $this->getRequestParams();
    $params += [
      'EmailAddress' => $email,
    ];

    return $this->set($params, 'GetVisitorWebserviceLoginKeyByEmail');

  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorOrderHistory(int $visitorId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
    ];

    return $this->get($params, 'GetVisitorOrderHistory');
  }

  /**
   * {@inheritdoc}
   */
  public function getAllVisitorIdsWhoOrderedLast48Hrs(): array {
    $params = $this->getRequestParams();

    return $this->get($params, 'GetAllVisitorIdsWhoOrderedLast48Hrs');
  }

  /**
   * {@inheritdoc}
   */
  public function getAllVisitorIdsWhoOrderedLastnMinutes(int $minutes): array {
    $params = $this->getRequestParams();

    // Limit minutes, which is restricted to 1440.
    if ($minutes > 1440) {
      $minutes = 1440;
    }

    $params += [
      'Minutes' => $minutes,
    ];

    return $this->get($params, 'GetAllVisitorIdsWhoOrderedLastnMinutes');
  }

  /**
   * {@inheritdoc}
   */
  public function eventAndShowsImport(): array {
    // @todo Implement eventAndShowsImport() method.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getPassList(): array {
    $params = $this->getRequestParams();

    return $this->get($params, 'GetPassList');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorPassesList(int $visitorId, \DateTime $from): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
      'FromCreationDate' => $from->format(self::DATEFORMAT),
    ];

    return $this->get($params, 'GetVisitorPassesList');
  }

  /**
   * {@inheritdoc}
   */
  public function savePassToVisitor(): array {
    // @todo Implement savePassToVisitor() method.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function removePassFromVisitor(): array {
    // @todo Implement removePassFromVisitor() method.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorCharacteristicsList(int $visitorId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
    ];

    return $this->get($params, 'GetVisitorCharacteristicsList');
  }

  /**
   * {@inheritdoc}
   */
  public function getCharacteristicsForVisitorList(): array {
    $params = $this->getRequestParams();

    return $this->get($params, 'GetCharacteristicsForVisitorList');
  }

  /**
   * {@inheritdoc}
   */
  public function saveCharacteristicToVisitor(int $visitorId, int $characteristicsId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
      'VisitorCharacteristicId' => $characteristicsId,
    ];

    return $this->set($params, 'SaveCharacteristicToVisitor');
  }

  /**
   * {@inheritdoc}
   */
  public function removeCharacteristicsFromVisitor(int $visitorId, int $characteristicsId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
      'VisitorCharacteristicId' => $characteristicsId,
    ];

    return $this->set($params, 'RemoveCharacteristicsFromVisitor');
  }

  /**
   * {@inheritdoc}
   */
  public function getVisitorInterestsList(int $visitorId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
    ];

    return $this->set($params, 'GetVisitorInterestsList');
  }

  /**
   * {@inheritdoc}
   */
  public function addInterestToVisitor(int $visitorId, int $interestId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
      'InterestId' => $interestId,
    ];

    return $this->set($params, 'AddInterestToVisitor');
  }

  /**
   * {@inheritdoc}
   */
  public function removeInterestsFromVisitor(int $visitorId, int $interestId): array {
    $params = $this->getRequestParams();
    $params += [
      'VisitorId' => $visitorId,
      'InterestId' => $interestId,
    ];

    return $this->set($params, 'RemoveInterestsFromVisitor');
  }

  /**
   * {@inheritdoc}
   */
  public function setVisitorNewsLetterByEmail(string $email, bool $newsletter): array {
    $params = $this->getRequestParams();
    $params += [
      'visitorEmail' => $email,
      'receivesNewsLetter' => $newsletter,
    ];

    return $this->set($params, 'SetVisitorNewsLetterByEmail');
  }

  /**
   * {@inheritdoc}
   */
  public function getRouteList(\DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();
    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
    ];

    return $this->get($params, 'GetRouteList');
  }

  /**
   * {@inheritdoc}
   */
  public function getRoutesWithShowsList(\DateTime $from, \DateTime $to): array {
    $params = $this->getRequestParams();
    $params += [
      'From' => $from->format(self::DATEFORMAT),
      'To' => $to->format(self::DATEFORMAT),
    ];

    return $this->get($params, 'GetRoutesWithShowsList');
  }

  /**
   * {@inheritDoc}
   */
  public function createVisitorIfNotExistsAndGetLoginKey(string $email): array {
    $params = $this->getMemberRequestParams();

    $params += [
      'emailAddress' => $email,
      'password' => 'test',
    ];

    return $this->set($params, 'CreateVisitorIfNotExistsAndGetLoginKey');
  }

}
