<?php

namespace Drupal\activetickets_client\Client;

/**
 * Interface class for ActiveTickets client.
 */
interface ActiveTicketsClientInterface {

  /**
   * Return config object.
   *
   * @return mixed
   *   Configuration object.
   */
  public function getConfig();

  /**
   * Initialize soapClient.
   *
   * @throws \Exception
   */
  public function initSoapClient();

  /**
   * Returns default request parameters.
   *
   * @return array
   *   Result array.
   */
  public function getRequestParams() : array;

  /**
   * Returns default request parameters.
   *
   * @return array
   *   Result array.
   */
  public function getMemberRequestParams() : array;

  /**
   * Parse the XML string returned by ActiveTickets.
   *
   * @param string $xml_string
   *   XML String.
   *
   * @return array
   *   Result array.
   */
  public function parseXmlResult(string $xml_string): array;

  /**
   * Call 'get' method from SOAP service.
   *
   * @param array $params
   *   Request parameters.
   * @param string $method
   *   Method name.
   *
   * @return array
   *   Result array.
   */
  public function get(array $params, string $method): array;

  /**
   * Call 'set' method from SOAP service.
   *
   * @param array $params
   *   Request parameters.
   * @param string $method
   *   Method name.
   *
   * @return array
   *   Result array.
   */
  public function set(array $params, string $method): array;

  /**
   * Get program list from cache.
   *
   * @param \DateTime $from
   *   From date.
   * @param \DateTime $to
   *   To date.
   * @param int $genreId
   *   Genre ID.
   *
   * @return array
   *   Program list.
   */
  public function getCachedProgramList(\DateTime $from, \DateTime $to, int $genreId): array;

  /**
   * Get program list.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   * @param int $genreId
   *   Genre ID.
   * @param string $csvKeywords
   *   Keywords.
   * @param bool $includePrices
   *   Include prices.
   *
   * @return array
   *   Result array.
   */
  public function getProgramList(\DateTime $from, \DateTime $to, int $genreId = 0, string $csvKeywords = '', bool $includePrices = TRUE);

  /**
   * Get deleted and inactive programs.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getProgramDeletedAndinActives(\DateTime $from, \DateTime $to) : array;

  /**
   * Get program list with custom key.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   * @param int $genreId
   *   Genre ID.
   * @param string $csvKeywords
   *   Keywords.
   * @param bool $includePrices
   *   Include prices.
   *
   * @return array
   *   Result array.
   */
  public function getProgramListWithCustomKey(\DateTime $from, \DateTime $to, int $genreId, string $csvKeywords, bool $includePrices);

  /**
   * Get program list with custom key IFFR.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   * @param int $genreId
   *   Genre ID.
   * @param string $csvKeywords
   *   Keywords.
   * @param bool $includePrices
   *   Include prices.
   *
   * @return array
   *   Result array.
   */
  public function getProgramListWithCustomKeyIffr(\DateTime $from, \DateTime $to, int $genreId, string $csvKeywords, bool $includePrices);

  /**
   * Get program list big.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   * @param int $genreId
   *   Genre ID.
   * @param string $csvKeywords
   *   Keywords.
   * @param bool $includePrices
   *   Include prices.
   *
   * @return array
   *   Result array.
   */
  public function getProgramListBig(\DateTime $from, \DateTime $to, int $genreId = 0, string $csvKeywords = '', bool $includePrices = TRUE);

  /**
   * Get program list with capacities.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   * @param int $genreId
   *   Genre ID.
   * @param string $csvKeywords
   *   Keywords.
   * @param bool $includePrices
   *   Include prices.
   * @param bool $includeCapacities
   *   Include capacities.
   *
   * @return array
   *   Result array.
   */
  public function getProgramListWithCapacities(\DateTime $from, \DateTime $to, int $genreId, string $csvKeywords, bool $includePrices, bool $includeCapacities);

  /**
   * Get program list with capacities and sold.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   * @param int $genreId
   *   Genre ID.
   * @param string $csvKeywords
   *   Keywords.
   * @param bool $includePrices
   *   Include prices.
   * @param bool $includeCapacities
   *   Include capacities.
   *
   * @return array
   *   Result array.
   */
  public function getProgramListWithCapacitiesAndSold(\DateTime $from, \DateTime $to, int $genreId, string $csvKeywords, bool $includePrices, bool $includeCapacities);

  /**
   * Get program capacities by rang.
   *
   * @return array
   *   Result array.
   */
  public function getProgramCapacitiesByRang();

  /**
   * Get genre list.
   *
   * @return array
   *   Result array.
   */
  public function getGenreList():array;

  /**
   * Get location list.
   *
   * @return array
   *   Result array.
   */
  public function getLocationList();

  /**
   * Get characteristics list.
   *
   * @return array
   *   Result array.
   */
  public function getCharacteristicsList();

  /**
   * Get subgenre list.
   *
   * @return array
   *   Result array.
   */
  public function getSubGenreList();

  /**
   * Get program ids via subgenre list.
   *
   * @param int $subGenreId
   *   Subgenre ID.
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getProgramIdsViaSubgenresList(int $subGenreId, \DateTime $from, \DateTime $to);

  /**
   * Get subgenres via program list.
   *
   * @param int $programId
   *   Program ID.
   *
   * @return array
   *   Result array.
   */
  public function getSubgenresViaProgramList(int $programId);

  /**
   * Get program detail.
   *
   * @param int $programId
   *   Program ID.
   *
   * @return array
   *   Result array.
   */
  public function getProgramDetail(int $programId);

  /**
   * Get program list light.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   * @param int $genreId
   *   Genre ID.
   * @param string $keywords
   *   Keywords.
   *
   * @return array
   *   Result array.
   */
  public function getProgramListLight(\DateTime $from, \DateTime $to, int $genreId, string $keywords);

  /**
   * Get program confirmed.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getProgramConfirmed(\DateTime $from, \DateTime $to);

  /**
   * Get program status.
   *
   * @param int $programId
   *   Program ID.
   *
   * @return array
   *   Result array.
   */
  public function getProgramStatus(int $programId): array;

  /**
   * Get program kijkwijzer.
   *
   * Requires special permission from ActiveTickets.
   *
   * @param int $programId
   *   Program ID.
   *
   * @return array
   *   Result array.
   */
  public function getProgramKijkwijzer(int $programId): array;

  /**
   * Get program pre-sale.
   *
   * Requires permission from ActiveTickets.
   *
   * @param int $programId
   *   Program ID.
   *
   * @return array
   *   Result array.
   */
  public function getProgramPresale(int $programId): array;

  /**
   * Get visitors.
   *
   * Requires permission from ActiveTickets.
   *
   * @return array
   *   Result array.
   */
  public function getVisitors(): array;

  /**
   * Get visitors within modification period.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorsWithinModificationPeriod(\DateTime $from, \DateTime $to): array;

  /**
   * Get visitors within modification period.
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorsWithinModificationPeriod2(\DateTime $from, \DateTime $to): array;

  /**
   * Get visitors light.
   *
   * Requires permission to access "GetAllVisitors".
   *
   * @return array
   *   Result array.
   */
  public function getVisitorsLight(): array;

  /**
   * Get visitors light within creation period.
   *
   * Requires permission to access "GetAllVisitorsFromCreationPeriod".
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorsLightWithinCreationPeriod(\DateTime $from, \DateTime $to): array;

  /**
   * Get visitors light within creation period.
   *
   * Requires permission to access "GetAllVisitorsFromCreationPeriod".
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorsLightWithinCreationPeriod2(\DateTime $from, \DateTime $to): array;

  /**
   * Get visitors within creation period.
   *
   * Requires permission to access "GetAllVisitorsFromCreationPeriod".
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorsWithinCreationPeriod(\DateTime $from, \DateTime $to): array;

  /**
   * Get visitors within creation period.
   *
   * Requires permission to access "GetAllVisitorsFromCreationPeriod".
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorsWithinCreationPeriod2(\DateTime $from, \DateTime $to): array;

  /**
   * Get visitor by id.
   *
   * @param int $visitorId
   *   Visitor ID.
   *
   * @return array
   *   Result array.
   */
  public function getVisitor(int $visitorId): array;

  /**
   * Get visitor by id.
   *
   * @param int $visitorId
   *   Visitor ID.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorBig(int $visitorId): array;

  /**
   * Get visitor by credentials.
   *
   * @param string $email
   *   E-mail address.
   * @param string $password
   *   Password.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorByCredentials(string $email, string $password): array;

  /**
   * Get visitor by credentials.
   *
   * @param string $email
   *   Email address.
   * @param string $password
   *   Password.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorByCredentials2(string $email, string $password): array;

  /**
   * Get visitor by email address.
   *
   * @param string $email
   *   Email address.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorByEmail(string $email): array;

  /**
   * Get visitor by email address.
   *
   * @param string $email
   *   Email address.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorByEmail2(string $email): array;

  /**
   * Get visitor by email address.
   *
   * @param string $email
   *   Email address.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorByEmailBig(string $email): array;

  /**
   * Set new visitor.
   *
   * @param string $email
   *   Email address.
   * @param string $password
   *   Password.
   * @param string $initials
   *   Initials.
   * @param string $firstname
   *   Firstname.
   * @param string $infix
   *   Infix.
   * @param string $lastname
   *   Lastname.
   * @param string $street
   *   Street.
   * @param string $housenumber
   *   Housenumber.
   * @param string $city
   *   City.
   * @param string $zipcode
   *   Zipcode.
   * @param int $countryId
   *   Country ID.
   * @param string $phoneNumber1
   *   Phone number.
   * @param string $phoneNumber2
   *   Second phonenumber.
   * @param \DateTime $dateOfBirth
   *   Date of birth.
   * @param string $gender
   *   Gender.
   * @param bool $onMailingList
   *   Mailing list.
   * @param string $bankAccount
   *   Bankaccount.
   * @param string $organisation
   *   Organisation.
   *
   * @return array
   *   Result array.
   */
  public function setVisitorNew(string $email, string $password, string $initials, string $firstname, string $infix, string $lastname, string $street, string $housenumber, string $city, string $zipcode, int $countryId, string $phoneNumber1, string $phoneNumber2, \DateTime $dateOfBirth, string $gender, bool $onMailingList, string $bankAccount, string $organisation): array;

  /**
   * Set new visitor.
   *
   * @param string $email
   *   Email address.
   * @param string $password
   *   Password.
   * @param string $initials
   *   Initials.
   * @param string $firstname
   *   Firstname.
   * @param string $infix
   *   Infix.
   * @param string $lastname
   *   Lastname.
   * @param string $street
   *   Street.
   * @param string $housenumber
   *   Housenumber.
   * @param string $city
   *   City.
   * @param string $zipcode
   *   Zipcode.
   * @param int $countryId
   *   Country ID.
   * @param string $phoneNumber1
   *   Phone number.
   * @param string $phoneNumber2
   *   Second phonenumber.
   * @param \DateTime $dateOfBirth
   *   Date of birth.
   * @param string $gender
   *   Gender.
   * @param bool $onMailingList
   *   Mailing list.
   * @param string $bankAccount
   *   Bankaccount.
   * @param string $organisation
   *   Organisation.
   *
   * @return array
   *   Result array.
   */
  public function setVisitorNewGender(string $email, string $password, string $initials, string $firstname, string $infix, string $lastname, string $street, string $housenumber, string $city, string $zipcode, int $countryId, string $phoneNumber1, string $phoneNumber2, \DateTime $dateOfBirth, string $gender, bool $onMailingList, string $bankAccount, string $organisation): array;

  /**
   * Delete visitor.
   *
   * @param int $visitorId
   *   Visitor Id.
   *
   * @return array
   *   Result array.
   */
  public function setVisitorDelete(int $visitorId): array;

  /**
   * Update visitor data.
   *
   * @param array $params
   *   Array of parameters.
   *
   * @return array
   *   Result array.
   */
  public function setVisitorUpdate(array $params): array;

  /**
   * Update visitor data.
   *
   * @param int $visitorId
   *   Visitor Id.
   * @param string $email
   *   Email address.
   * @param string $password
   *   Password.
   * @param string $initials
   *   Initials.
   * @param string $firstname
   *   Firstname.
   * @param string $infix
   *   Infix.
   * @param string $lastname
   *   Lastname.
   * @param string $street
   *   Street.
   * @param string $housenumber
   *   Housenumber.
   * @param string $city
   *   City.
   * @param string $zipcode
   *   Zipcode.
   * @param int $countryId
   *   Country ID.
   * @param string $phoneNumber1
   *   Phone number.
   * @param string $phoneNumber2
   *   Second phonenumber.
   * @param \DateTime $dateOfBirth
   *   Date of birth.
   * @param string $gender
   *   Gender.
   * @param bool $onMailingList
   *   Mailing list.
   * @param string $bankAccount
   *   Bankaccount.
   * @param string $organisation
   *   Organisation.
   *
   * @return array
   *   Result array.
   */
  public function setVisitorUpdateGender(int $visitorId, string $email, string $password, string $initials, string $firstname, string $infix, string $lastname, string $street, string $housenumber, string $city, string $zipcode, int $countryId, string $phoneNumber1, string $phoneNumber2, \DateTime $dateOfBirth, string $gender, bool $onMailingList, string $bankAccount, string $organisation): array;

  /**
   * Get login key.
   *
   * @param int $visitorId
   *   Visitor Id.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorWebserviceLoginKey(int $visitorId): array;

  /**
   * Get login key by email.
   *
   * @param string $email
   *   Email address.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorWebserviceLoginKeyByEmail(string $email): array;

  /**
   * Get visitor order history.
   *
   * @param int $visitorId
   *   Visitor Id.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorOrderHistory(int $visitorId): array;

  /**
   * Get visitors ids who ordered in the last 48 hours.
   *
   * @return array
   *   Result array.
   */
  public function getAllVisitorIdsWhoOrderedLast48Hrs(): array;

  /**
   * Get visitor ids who ordered in the last N minutes.
   *
   * @param int $minutes
   *   Number of minutes (max 1440).
   *
   * @return array
   *   Result array.
   */
  public function getAllVisitorIdsWhoOrderedLastnMinutes(int $minutes): array;

  /**
   * Import events and shows.
   *
   * @return array
   *   Result array.
   */
  public function eventAndShowsImport(): array;

  /**
   * Get pass list.
   *
   * Requires permission 'GetPassList'.
   *
   * @return array
   *   Result array.
   */
  public function getPassList(): array;

  /**
   * Get visitor passes list.
   *
   * Requires permission 'GetPassList'.
   *
   * @param int $visitorId
   *   Visitor Id.
   * @param \DateTime $from
   *   From creation date.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorPassesList(int $visitorId, \DateTime $from): array;

  /**
   * Save password to visitor.
   *
   * @return array
   *   Result array.
   */
  public function savePassToVisitor(): array;

  /**
   * Remove password from visitor.
   *
   * @return array
   *   Result array.
   */
  public function removePassFromVisitor(): array;

  /**
   * Get visitor characteristics list.
   *
   * @param int $visitorId
   *   Visitor Id.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorCharacteristicsList(int $visitorId): array;

  /**
   * Get characteristics list.
   *
   * @return array
   *   Result array.
   */
  public function getCharacteristicsForVisitorList(): array;

  /**
   * Save characteristic to visitor.
   *
   * @param int $visitorId
   *   Visitor Id.
   * @param int $characteristicId
   *   Characteristic Id.
   *
   * @return array
   *   Result array.
   */
  public function saveCharacteristicToVisitor(int $visitorId, int $characteristicId): array;

  /**
   * Remove characteristic from visitor.
   *
   * @param int $visitorId
   *   Visitor Id.
   * @param int $characteristicId
   *   Characteristic Id.
   *
   * @return array
   *   Result array.
   */
  public function removeCharacteristicsFromVisitor(int $visitorId, int $characteristicId): array;

  /**
   * Get visitor interest list.
   *
   * @param int $visitorId
   *   Visitor Id.
   *
   * @return array
   *   Result array.
   */
  public function getVisitorInterestsList(int $visitorId): array;

  /**
   * Add interest to visitor.
   *
   * @param int $visitorId
   *   Visitor Id.
   * @param int $interestId
   *   Interest Id.
   *
   * @return array
   *   Result array.
   */
  public function addInterestToVisitor(int $visitorId, int $interestId): array;

  /**
   * Remove interest from visitor.
   *
   * @param int $visitorId
   *   Visitor Id.
   * @param int $interestId
   *   Interest Id.
   *
   * @return array
   *   Result array.
   */
  public function removeInterestsFromVisitor(int $visitorId, int $interestId): array;

  /**
   * Set newsletter flag by email.
   *
   * @param string $email
   *   Email address.
   * @param bool $newsletter
   *   Receives newsletter.
   *
   * @return array
   *   Result array.
   */
  public function setVisitorNewsLetterByEmail(string $email, bool $newsletter): array;

  /**
   * Get route list.
   *
   * Requires permission to access "GetRoutes".
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getRouteList(\DateTime $from, \DateTime $to): array;

  /**
   * Get route list.
   *
   * Requires permission to access "GetRoutes".
   *
   * @param \DateTime $from
   *   Start date.
   * @param \DateTime $to
   *   End date.
   *
   * @return array
   *   Result array.
   */
  public function getRoutesWithShowsList(\DateTime $from, \DateTime $to): array;

  /**
   * Create visitor.
   *
   * @param string $email
   *   Email address.
   *
   * @return array
   *   Return data.
   */
  public function createVisitorIfNotExistsAndGetLoginKey(string $email): array;

}
