<?php

namespace Drupal\acumatica\Plugin\EntitySync\FieldTransformer;

use Drupal\acumatica\Acumatica\Model\AttributeHelper;
use Drupal\acumatica\Exception\MissingAttribute;
use Drupal\entity_sync\FieldTransformer\PluginBase;
use Drupal\Core\Entity\ContentEntityInterface;

/**
 * Transformer that imports the value of an Acumatica attribute.
 *
 * Supported configuration properties:
 * - allow_missing_attribute: (boolean, optional, defaults to `FALSE`)
 *   When set to `FALSE` (default), an exception will be raised when the
 *   requested attribute is not found. When set to `TRUE` and the requested
 *   attribute is not found, the transformed value will be `NULL.`
 *
 * So far, simple attributes are supported which have the value stored in a
 * field called `Value` with one main property called `value`. More complex
 * attributes with multiple properties - if they exist - could be supported here
 * in the future.
 *
 * @EntitySyncFieldTransformer(
 *   id = "acumatica_attribute"
 * )
 */
class Attribute extends PluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'allow_missing_attribute' => FALSE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  protected function transformExportedValue(
    $value,
    ContentEntityInterface $local_entity,
    $remote_entity_id,
    array $field_info,
    array $context
  ) {
    return ['value' => $value];
  }

  /**
   * {@inheritdoc}
   */
  protected function transformImportedValue(
    $value,
    \stdClass $remote_entity,
    ?ContentEntityInterface $local_entity,
    array $field_info,
    array $context
  ) {
    // All the logic is on the helper method for reusability. This method would
    // raise errors if the attribute does not exist on the object. It seems that
    // if the attribute is defined on the object's class, the attribute will
    // exist on the object (if it is properly requested in the API call that
    // attributes are expanded) but its Value field will be empty. However, we
    // have found cases where this does not happen i.e. the attribute does not
    // exist on the object for some items while it does on other. Thus, in
    // general, we want to let the errors be raised as it would be an error with
    // the API calls not requesting the attributes to be expanded allow
    // configuring whether to raise an error or not; but, to not raise an error
    // when configured so using `allow_missing_attribute` to handle special
    // cases.

    try {
      return AttributeHelper::getValue(
        $field_info['remote_name'],
        $remote_entity
      );
    } catch (MissingAttribute $exception) {
      if ($this->configuration['allow_missing_attribute']) {
        return NULL;
      }
      throw $exception;
    }
  }

}
