<?php

namespace Drupal\ad_content;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityHandlerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines an access control handler for ad contents.
 */
class AdContentAccessControlHandler extends EntityAccessControlHandler implements EntityHandlerInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a AdContentAccessControlHandler object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($entity_type);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    /** @var \Drupal\ad_content\Entity\AdContentInterface $entity */
    // Allow admin permission to override all operations.
    if ($account->hasPermission($this->entityType->getAdminPermission())) {
      return AccessResult::allowed()->cachePerPermissions();
    }

    $type = $entity->bundle();
    $editPermissions = ['edit ' . $type . ' ads', 'edit ads'];
    $deletePermissions = ['delete ' . $type . ' ads', 'delete ads'];
    $viewRevisionsPermissions = ['view ' . $type . ' ad revisions', 'view ad revisions'];
    $revertRevisionsPermissions = ['revert ' . $type . ' ad revisions', 'revert ad revisions'];
    $deleteRevisionsPermissions = ['delete ' . $type . ' ad revisions', 'delete ad revisions'];

    switch ($operation) {
      case 'update':
        return AccessResult::allowedIfHasPermissions($account, $editPermissions, 'OR')->cachePerPermissions();

      case 'delete':
        return AccessResult::allowedIfHasPermissions($account, $deletePermissions, 'OR')->cachePerPermissions();

      case 'view all revisions':
      case 'view revision':
        return AccessResult::allowedIfHasPermissions($account, $viewRevisionsPermissions, 'OR')->cachePerPermissions();

      case 'revert':
        return AccessResult::allowedIfHasPermissions($account, $revertRevisionsPermissions, 'OR')->cachePerPermissions();

      case 'delete revision':
        return AccessResult::allowedIfHasPermissions($account, $deleteRevisionsPermissions, 'OR')->cachePerPermissions();

      default:
        return AccessResult::neutral()->cachePerPermissions();
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    $permissions = [
      'administer ads',
      'create ads',
      'create ' . $entity_bundle . ' ads',
    ];
    return AccessResult::allowedIfHasPermissions($account, $permissions, 'OR');
  }

}
