<?php

namespace Drupal\ad_content\Entity;

use Drupal\ad_content\AdContentAccessControlHandler;
use Drupal\ad_content\Form\AdContentEntityForm;
use Drupal\Core\Url;
use Drupal\user\EntityOwnerTrait;
use Drupal\views\EntityViewsData;
use Drupal\Core\Entity\EntityViewBuilder;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\EditorialContentEntityBase;
use Drupal\Core\Entity\Form\RevisionDeleteForm;
use Drupal\Core\Entity\Form\RevisionRevertForm;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Entity\Routing\RevisionHtmlRouteProvider;

/**
 * Defines the advertisement entity class.
 */
#[ContentEntityType(
  id:'ad_content',
  label: new TranslatableMarkup('Advertisement'),
  label_collection: new TranslatableMarkup('Advertisements'),
  label_singular: new TranslatableMarkup('Advertisement'),
  label_plural: new TranslatableMarkup('Advertisements'),
  label_count: [
    'singular' => '@count advertisement',
    'plural' => '@count advertisements',
  ],
  handlers: [
    'storage' => SqlContentEntityStorage::class,
    'view_builder' => EntityViewBuilder::class,
    'list_builder' => AdContentListBuilder::class,
    'access' => AdContentAccessControlHandler::class,
    'form' => [
      'default' => AdContentEntityForm::class,
      'add' => AdContentEntityForm::class,
      'edit' => AdContentEntityForm::class,
      'delete' => ContentEntityDeleteForm::class,
      'delete-multiple-confirm' => DeleteMultipleForm::class,
      'revision-delete' => RevisionDeleteForm::class,
      'revision-revert' => RevisionRevertForm::class,
    ],
    'views_data' => EntityViewsData::class,
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
      'revision' => RevisionHtmlRouteProvider::class,
    ],
  ],
  base_table: 'ad_content',
  data_table: 'ad_content_field_data',
  revision_table: 'ad_content_revision',
  revision_data_table: 'ad_content_field_revision',
  show_revision_ui: TRUE,
  permission_granularity: 'bundle',
  admin_permission: 'administer ads',
  bundle_entity_type: 'ad_content_type',
  bundle_label: new TranslatableMarkup('Advertisement type'),
  field_ui_base_route: 'entity.ad_content_type.edit_form',
  links: [
    'add-page' => '/admin/content/ad/add',
    'add-form' => '/admin/content/ad/add/{ad_content_type}',
    'canonical' => '/admin/content/ad/{ad_content}',
    'collection' => '/admin/content/ad',
    'delete-form' => '/admin/content/ad/{ad_content}/delete',
    'delete-multiple-form' => '/media/delete',
    'edit-form' => '/admin/content/ad/{ad_content}/edit',
    'revision' => '/admin/content/ad/{ad_content}/revisions/{ad_content_revision}/view',
    'revision-delete-form' => '/admin/content/ad/{ad_content}/revision/{ad_content_revision}/delete',
    'revision-revert-form' => '/admin/content/ad/{ad_content}/revision/{ad_content_revision}/revert',
    'version-history' => '/admin/content/ad/{ad_content}/revisions',
  ],
  entity_keys: [
    'id' => 'id',
    'bundle' => 'bundle',
    'revision' => 'revision_id',
    'label' => 'title',
    'uuid' => 'uuid',
    'published' => 'status',
    'owner' => 'uid',
    'langcode' => 'langcode',
  ],
  revision_metadata_keys: [
    'revision_user' => 'revision_user',
    'revision_created' => 'revision_created',
    'revision_log_message' => 'revision_log_message',
  ],
  render_cache: TRUE,
  translatable: TRUE,
  common_reference_target: FALSE,
)]
class AdContent extends EditorialContentEntityBase implements AdContentInterface {

  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function getAdIdentifier(): string {
    return $this->uuid();
  }

  /**
   * {@inheritdoc}
   */
  public function getBucketId(): string {
    return static::BUCKET_ID;
  }

  /**
   * Returns the ad placement ID.
   *
   * @return string
   *   The ad placement ID.
   */
  public function getPlacementId(): string {
    return $this->get('placement')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetUrl(): ?Url {
    $value = current($this->get('target_url')->getValue());
    $value['options']['attributes']['rel'] = 'nofollow';
    return $value ? Url::fromUri($value['uri'], $value['options']) : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    /** @var \Drupal\Core\Field\BaseFieldDefinition[] $fields */
    $fields = parent::baseFieldDefinitions($entity_type);
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    $fields['status']
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => 15,
      ]);

    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Title'))
      ->setDescription(new TranslatableMarkup('A brief description of the advertisement.'))
      ->setRevisionable(TRUE)
      ->setRequired(TRUE)
      ->addConstraint('UniqueField', [])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setTranslatable(TRUE);

    $fields['uid']
      ->setLabel(new TranslatableMarkup('Authored by'))
      ->setDescription(t('The user ID of the author.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['placement'] = BaseFieldDefinition::create('list_string')
      ->setLabel(new TranslatableMarkup('Placement'))
      ->setDescription(new TranslatableMarkup("The advertisement's placement."))
      ->setRevisionable(TRUE)
      ->setRequired(TRUE)
      ->setSettings([
        'allowed_values_function' => 'ad_get_placements_list',
      ])
      ->setDisplayOptions('view', [
        'type' => 'list_default',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['target_url'] = BaseFieldDefinition::create('link')
      ->setLabel(new TranslatableMarkup('Target URL'))
      ->setDescription(new TranslatableMarkup('The URL to be taken to when clicking on the advertisement.'))
      ->setRevisionable(TRUE)
      ->setRequired(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'link_default',
        'weight' => -3,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'link',
        'weight' => -3,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setTranslatable(TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(new TranslatableMarkup('Authored on'))
      ->setDescription(t('The time that the advertisement was created.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'timestamp',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the advertisement was last edited.'));

    return $fields;
  }

}
