<?php

namespace Drupal\ad_content;

use Drupal\ad_content\Entity\AdContentType;
use Drupal\Core\Entity\BundlePermissionHandlerTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides dynamic permissions for ads of different types.
 */
class AdContentPermissions {

  use BundlePermissionHandlerTrait;
  use StringTranslationTrait;

  /**
   * Returns an array of ad type permissions.
   *
   * @return array
   *   The ad type permissions.
   *
   * @see \Drupal\user\PermissionHandlerInterface::getPermissions()
   */
  public function contentAdTypePermissions() {
    return $this->generatePermissions(
          AdContentType::loadMultiple(), [
            $this,
            'buildPermissions',
          ]
      );
  }

  /**
   * Returns a list of AdContent permissions for a given AdContent type.
   *
   * @param \Drupal\ad_content\Entity\AdContentType $type
   *   The AdContent type.
   *
   * @return array
   *   An associative array of permission names and descriptions.
   */
  protected function buildPermissions(AdContentType $type) {
    $typeId = $type->id();
    $type_params = ['%type_name' => $type->label()];

    return [
      "create $typeId ads" => [
        'title' => $this->t('%type_name: Create advertisements', $type_params),
      ],
      "edit $typeId ads" => [
        'title' => $this->t('%type_name: Edit advertisements', $type_params),
      ],
      "delete $typeId ads" => [
        'title' => $this->t('%type_name: Delete advertisements', $type_params),
      ],
      "view $typeId ad revisions" => [
        'title' => $this->t('%type_name: View revisions', $type_params),
      ],
      "revert $typeId ad revisions" => [
        'title' => $this->t('%type_name: Revert revisions', $type_params),
      ],
      "delete $typeId ad revisions" => [
        'title' => $this->t('%type_name: Delete revisions', $type_params),
      ],
    ];
  }

}
