<?php

namespace Drupal\ad_content\Controller;

use Drupal\ad\Bucket\BucketFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Render\RendererInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Ad impression controller.
 */
class ImpressionController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * The ad bucket factory.
   *
   * @var \Drupal\ad\Bucket\BucketFactoryInterface
   */
  protected BucketFactoryInterface $bucketFactory;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * Impression constructor.
   *
   * @param \Drupal\ad\Bucket\BucketFactoryInterface $bucket_factory
   *   The ad bucket factory.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(
    BucketFactoryInterface $bucket_factory,
    RendererInterface $renderer,
  ) {
    $this->bucketFactory = $bucket_factory;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ad.bucket_factory'),
      $container->get('renderer'),
    );
  }

  /**
   * Renders the specified ads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function renderAds(Request $request): JsonResponse {
    $response_data = [];
    $request_data = $request->query->all();

    if (!empty($request_data['ads'])) {
      foreach ($request_data['ads'] as $html_id => $data) {
        $bucket = $this->bucketFactory->get($data['bucket']);
        $build = $bucket->buildAd($data['placement']);
        $response_data[$html_id] = $this->renderer->renderInIsolation($build);
      }
    }

    return new JsonResponse($response_data);
  }

}
