<?php

namespace Drupal\ad_content\Entity;

use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\ad_content\Form\AdContentTypeEntityForm;
use Drupal\Core\Config\Entity\ConfigEntityBundleBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;

/**
 * Defines the advertisement type bundle class.
 */
#[ConfigEntityType(
  id: 'ad_content_type',
  label: new TranslatableMarkup('Advertisement Type'),
  label_singular: new TranslatableMarkup('Advertisement type'),
  label_plural: new TranslatableMarkup('Advertisement types'),
  label_count: [
    'singular' => '@count advertisement type',
    'plural' => '@count advertisement types',
  ],
  bundle_of: 'ad_content',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  config_prefix: 'ad_content_type',
  config_export: [
    'id',
    'label',
    'description',
  ],
  handlers: [
    'list_builder' => AdContentTypeListBuilder::class,
    'form' => [
      'default' => AdContentTypeEntityForm::class,
      'add' => AdContentTypeEntityForm::class,
      'edit' => AdContentTypeEntityForm::class,
      'delete' => EntityDeleteForm::class,
    ],
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  admin_permission: 'administer ads',
  links: [
    'canonical' => '/admin/structure/ad-content/{ad_content_type}',
    'add-form' => '/admin/structure/ad-content/add',
    'edit-form' => '/admin/structure/ad-content/manage/{ad_content_type}',
    'delete-form' => '/admin/structure/ad-content/manage/{ad_content_type}/delete',
    'collection' => '/admin/structure/ad-content',
  ]
)]
class AdContentType extends ConfigEntityBundleBase implements AdContentTypeInterface {

  /**
   * A brief description of the advertisement type.
   *
   * @var string
   */
  protected $description;

  /**
   * {@inheritDoc}
   */
  public function getDescription(): string {
    return $this->description ?? '';
  }

  /**
   * {@inheritDoc}
   */
  public function setDescription($description) {
    return $this->set('description', $description);
  }

  /**
   * Returns the UUID of the advertisement type.
   */
  public function getUuid(): string {
    return $this->uuid();
  }

  /**
   * Returns the id of the advertisement type.
   */
  public function getId(): string {
    return $this->id();
  }

  /**
   * Returns the label of the advertisement type.
   */
  public function getLabel(): string {
    return $this->label();
  }

}
