<?php

namespace Drupal\ad_content\Plugin\Ad\Bucket;

use Drupal\Core\Utility\Error;
use Drupal\ad\AdInterface;
use Drupal\ad\Bucket\BucketInterface;
use Drupal\ad\Track\TrackerFactoryInterface;
use Drupal\ad\Track\TrackerInterface;
use Drupal\ad_content\Entity\AdContentInterface;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Component\Render\MarkupInterface;
use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Render\Markup;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\Session\AccountInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Basic advertisement bucket.
 *
 * @Plugin(
 *   id = \Drupal\ad_content\Entity\AdContentInterface::BUCKET_ID,
 *   label = @Translation("Advertisement"),
 * )
 */
class AdContentBucket extends PluginBase implements BucketInterface, ContainerFactoryPluginInterface, TrustedCallbackInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected EntityRepositoryInterface $entityRepository;

  /**
   * The ad tracker factory.
   *
   * @var \Drupal\ad\Track\TrackerFactoryInterface
   */
  protected TrackerFactoryInterface $trackerFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * AdContentBucket constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\ad\Track\TrackerFactoryInterface $tracker_factory
   *   The ad tracker factory.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    EntityRepositoryInterface $entity_repository,
    TrackerFactoryInterface $tracker_factory,
    AccountInterface $current_user,
    LoggerInterface $logger,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->entityRepository = $entity_repository;
    $this->trackerFactory = $tracker_factory;
    $this->currentUser = $current_user;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity.repository'),
      $container->get('ad.tracker_factory'),
      $container->get('current_user'),
      $container->get('logger.factory')->get('ad_content')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function isEmpty(string $placement_id): bool {
    $adContent = $this->getAdContent($placement_id, FALSE);
    return empty($adContent);
  }

  /**
   * {@inheritDoc}
   */
  public function getPlacementCount(string $placement_id): int {
    /** @var \Drupal\Core\Entity\ContentEntityStorageInterface $storage */
    $storage = $this->entityTypeManager->getStorage('ad_content');
    return $storage->getQuery()
      ->accessCheck()
      ->count()
      ->condition('placement', $placement_id)
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function getAd(string $id): ?AdInterface {
    $ad_content = NULL;

    try {
      /** @var \Drupal\ad_content\Entity\AdContentInterface $ad_content */
      $ad_content = $this->entityRepository->loadEntityByUuid('ad_content', $id);
    }
    catch (EntityStorageException $e) {
      Error::logException($this->logger, $e);
    }

    return $ad_content;
  }

  /**
   * {@inheritdoc}
   */
  public function getTracker(): TrackerInterface {
    return $this->trackerFactory->get($this->configuration['tracker_id']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildPlaceholder(string $placement_id): array {
    $bucket_id = $this->getPluginDefinition()['id'];
    $html_id = $bucket_id . '-' . $placement_id;

    return [
      'placeholder' => [
        '#type' => 'html_tag',
        '#tag' => 'ad-content',
        '#attributes' => [
          'id' => Html::getUniqueId($html_id),
          'bucket' => $bucket_id,
          'placement' => $placement_id,
          'data-nosnippet' => TRUE,
        ],
        '#attached' => [
          'library' => ['ad_content/ad_content.render_ads'],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildAd(string $placement_id): array {
    $build = [];

    $ad_content = $this->getAdContent($placement_id);
    if ($ad_content) {
      $build = $this->entityTypeManager
        ->getViewBuilder($ad_content->getEntityTypeId())
        ->view($ad_content, 'impression');

      $impression_id = $this->getTracker()
        ->trackImpression($ad_content, $this->currentUser, $this->configuration['ad_context']);

      $build['#ad_impression_id'] = $impression_id;
      $build['#post_render'][] = static::class . '::postAdRender';
    }

    return $build;
  }

  /**
   * Returns an ad content to be rendered.
   *
   * @param string $placement_id
   *   The ID of the ad placement.
   * @param bool $random
   *   When TRUE, the ad will be selected randomly.
   *
   * @return \Drupal\ad_content\Entity\AdContentInterface
   *   An advertisement entity.
   */
  protected function getAdContent(string $placement_id, bool $random = TRUE): ?AdContentInterface {
    $ad_content = NULL;

    try {
      $storage = $this->entityTypeManager->getStorage('ad_content');
      $query = $this->getAdQuery($placement_id)->accessCheck(TRUE);
      if ($random) {
        $query->addTag('order_random');
      }

      $result = $query->range(0, 1)->execute();
      if ($result) {
        $ad_content = $storage->load(reset($result));
      }
    }
    catch (PluginException $e) {
      Error::logException($this->logger, $e);
    }

    return $ad_content;
  }

  /**
   * Post render callback for the ad view builder.
   */
  public static function postAdRender($markup, array $build) {
    $output = str_replace(TrackerInterface::PLACEHOLDER_IMPRESSION, $build['#ad_impression_id'], $markup);
    return $markup instanceof MarkupInterface ? Markup::create($output) : $output;
  }

  /**
   * Retrieves a random ad with the specified placement.
   *
   * @param string $placement_id
   *   The ID of the ad placement.
   *
   * @return \Drupal\ad_content\Entity\AdContentInterface|null
   *   An ad entity or NULL if none could be found.
   */
  protected function getRandomAd(string $placement_id): ?AdContentInterface {
    $ad_content = NULL;

    try {
      /** @var \Drupal\Core\Entity\ContentEntityStorageInterface $storage */
      $storage = $this->entityTypeManager->getStorage('ad_content');

      $query = $this->getAdQuery($placement_id)
        ->accessCheck(TRUE);

      $result = $query
        ->addTag('order_random')
        ->range(0, 1)
        ->execute();

      if ($result) {
        /** @var \Drupal\ad_content\Entity\AdContentInterface $ad_content */
        $ad_content = $storage->load(reset($result));
      }
    }
    catch (PluginException $e) {
      Error::logException($this->logger, $e);
    }

    return $ad_content;
  }

  /**
   * Get the base query for finding an ad by placement.
   *
   * @param string $placement_id
   *   The ad placement ID.
   *
   * @return \Drupal\Core\Entity\Query\QueryInterface
   *   Ad base query.
   */
  protected function getAdQuery(string $placement_id): QueryInterface {
    /** @var \Drupal\Core\Entity\ContentEntityStorageInterface $storage */
    $storage = $this->entityTypeManager->getStorage('ad_content');

    return $storage->getQuery()
      ->accessCheck()
      ->condition('placement', $placement_id)
      ->condition('status', 1);
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return [
      'postAdRender',
    ];
  }

}
