<?php

namespace Drupal\ad_content\Entity;

use Drupal\ad\Entity\AdPlacementManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Advertisement list builder.
 */
class AdContentListBuilder extends EntityListBuilder {

  /**
   * The entity type manager service.
   *
   * @var Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The ad placement factory.
   */
  protected AdPlacementManagerInterface $placementFactory;

  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage, EntityTypeManagerInterface $entity_type_manager, AdPlacementManagerInterface $placement_factory) {
    parent::__construct($entity_type, $storage);
    $this->entityTypeManager = $entity_type_manager;
    $this->placementFactory = $placement_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('entity_type.manager'),
      $container->get('ad.placement_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header = [
      'id' => $this->t('ID'),
      'title' => $this->t('Title'),
      'placement' => $this->t('Placement'),
      'bundle' => $this->t('Type'),
      'status' => $this->t('Status'),
    ];

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $ad_content) {
    /** @var \Drupal\ad_content\Entity\AdContentInterface $ad_content */

    $row = [
      'id' => $ad_content->id(),
      'title' => [
        'data' => [
          '#type' => 'link',
          '#title' => $ad_content->label(),
          '#url' => $ad_content->toUrl(),
        ],
      ],
      'placement' => $this->placementFactory->get($ad_content->getPlacementId())->label(),
      'bundle' => $this->getBundleName($ad_content->bundle()),
      'status' => $ad_content->isPublished() ? $this->t('Published') : $this->t('Not published'),
    ];

    return $row + parent::buildRow($ad_content);
  }

  /**
   * Returns the name of the bundle with the given ID.
   *
   * @param string $bundleId
   *   The ID of the bundle.
   *
   * @return string
   *   The name of the bundle.
   */
  private function getBundleName($bundleId): string {
    $bundle = $this->entityTypeManager->getStorage('ad_content_type')->load($bundleId);
    return $bundle !== NULL ? $bundle->label() : $this->t('@bundleId (Not found)', ['@bundleId' => $bundleId]);
  }

}
