<?php

namespace Drupal\ad_content_scheduler\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines dynamic local tasks.
 *
 * The local tasks that define tabs for the 'Scheduled' entity views cannot be
 * hard-coded in the links.task.yml file because if a view is disabled its route
 * will not exist and this produces an exception "Route X does not exist." The
 * routes are defined here instead to enable checking that the views are loaded.
 */
class DynamicLocalTasks extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Creates a DynamicLocalTasks object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {

    $view_storage = $this->entityTypeManager->getStorage('view');

    $view = $view_storage->load('ad_content_scheduler_scheduled_ad_content');
    if ($view && $view->status() && $view->getDisplay('overview')) {
      // This task is added so that we get an 'overview' sub-task link alongside
      // the 'scheduled ad_content' sub-task link.
      $this->derivatives['ad_content_scheduler.ad_content_overview'] = [
        'title' => $this->t('Overview'),
        'route_name' => 'entity.ad_content.collection',
        'parent_id' => 'entity.ad_content.collection',
      ] + $base_plugin_definition;

      // Define local task for scheduled ad_content view.
      $this->derivatives['ad_content_scheduler.scheduled_ad_content'] = [
        'title' => $this->t('Scheduled Advertisements'),
        'route_name' => 'view.ad_content_scheduler_scheduled_ad_content.overview',
        'parent_id' => 'entity.ad_content.collection',
        'weight' => 10,
      ] + $base_plugin_definition;
    }

    return parent::getDerivativeDefinitions($base_plugin_definition);
  }

}
