<?php

declare(strict_types=1);

namespace Drupal\ad\Entity;

use Drupal\ad\Form\AdPlacementDeleteForm;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ad\Form\AdPlacementForm;

/**
 * Defines the ad placement entity type.
 */
#[ConfigEntityType(
  id: 'ad_placement',
  label: new TranslatableMarkup('Advertisement placement'),
  label_collection: new TranslatableMarkup('Advertisement placements'),
  label_singular: new TranslatableMarkup('advertisement placement'),
  label_plural: new TranslatableMarkup('advertisement placements'),
  config_prefix: 'ad_placement',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => AdPlacementListBuilder::class,
    'form' => [
      'add' => AdPlacementForm::class,
      'edit' => AdPlacementForm::class,
      'delete' => AdPlacementDeleteForm::class,
    ],
  ],
  links: [
    'collection' => '/admin/config/content/ad-placement',
    'add-form' => '/admin/config/content/ad-placement/add',
    'edit-form' => '/admin/config/content/ad-placement/{ad_placement}',
    'delete-form' => '/admin/config/content/ad-placement/{ad_placement}/delete',
  ],
  admin_permission: 'administer ads',
  label_count: [
    'singular' => '@count advertisement placement',
    'plural' => '@count advertisement placements',
  ],
  config_export: [
    'id',
    'label',
    'description',
  ],
)]
final class AdPlacement extends ConfigEntityBase implements AdPlacementInterface {

  /**
   * The ID of the ad placement.
   */
  protected string $id;

  /**
   * The label of the ad placement.
   */
  protected string $label;

  /**
   * An internal description of the ad placement.
   */
  protected string $description;

  /**
   * {@inheritDoc}
   */
  public function getDescription(): string {
    return $this->description;
  }

  /**
   * {@inheritDoc}
   */
  public function setDescription($description) {
    $this->description = $description;
  }

}
