<?php

namespace Drupal\ad\Form;

use Drupal\ad\Bucket\BucketFactoryInterface;
use Drupal\ad\Plugin\Ad\Track\NullTracker;
use Drupal\ad\Track\TrackerFactoryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Ad settings configuration form.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The ad bucket factory.
   *
   * @var \Drupal\ad\Bucket\BucketFactoryInterface
   */
  protected BucketFactoryInterface $bucketFactory;

  /**
   * The ad tracker factory.
   *
   * @var \Drupal\ad\Track\TrackerFactoryInterface
   */
  protected TrackerFactoryInterface $trackerFactory;

  /**
   * AdSettingsForm constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\ad\Bucket\BucketFactoryInterface $bucket_factory
   *   The ad bucket factory.
   * @param \Drupal\ad\Track\TrackerFactoryInterface $tracker_factory
   *   The ad tracker factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface|null $typedConfigManager
   *   The typed config manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    BucketFactoryInterface $bucket_factory,
    TrackerFactoryInterface $tracker_factory,
    ?TypedConfigManagerInterface $typedConfigManager,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
    $this->bucketFactory = $bucket_factory;
    $this->trackerFactory = $tracker_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('ad.bucket_factory'),
      $container->get('ad.tracker_factory'),
      $container->get('config.typed') ?? NULL
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ad_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ad.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ad.settings');
    $trackerSettings = $config->get('trackers');
    $trackers = $this->trackerFactory->getList();
    $buckets = $this->bucketFactory->getList();

    $form['trackers'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Tracker configuration'),
      '#description' => $this->t('Assign an advertisement statistics tracker for each advertisement source.'),
      '#tree' => TRUE,
    ];

    if (!$buckets || !array_diff_key($trackers, [NullTracker::TRACKER_ID => TRUE])) {
      $args = [
        '@url' => Url::fromRoute('system.modules_list')
          ->toString(TRUE)
          ->getGeneratedUrl(),
      ];
      $form['trackers']['#description'] = $this->t('You need to <a href="@url">enable</a> at least one advertisement source and one advertisement statistics tracker engine.', $args);
    }

    foreach ($buckets as $id => $label) {
      $form['trackers'][$id] = [
        '#type' => 'select',
        '#title' => $label,
        '#required' => TRUE,
        '#options' => $trackers,
        '#default_value' => $trackerSettings[$id] ?? key($trackers),
      ];
    }

    $form['advertisement_indicator'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Advertisement indicator'),
      '#description' => $this->t('In many countries it is mandatory to label advertisements as being such. This setting adds a small print (e.g. "Advertisement") to all advertisement blocks.'),
      '#default_value' => $config->get('advertisement_indicator') ?? '',
    ];

    $form['hide_empty_blocks'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide empty advertisement blocks'),
      '#description' => $this->t('Hide advertisement blocks entirely (including their placeholder) if they have no ads available.'),
      '#default_value' => $config->get('hide_empty_blocks'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('ad.settings')
      ->set('trackers', $form_state->getValue('trackers'))
      ->set('advertisement_indicator', $form_state->getValue('advertisement_indicator'))
      ->set('hide_empty_blocks', $form_state->getValue('hide_empty_blocks'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
