<?php

namespace Drupal\ad\Plugin\Block;

use Drupal\ad\Entity\AdPlacementManager;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ad\Bucket\BucketFactoryInterface;
use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Element\Checkboxes;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the ad slot block type.
 */
#[Block(
  id: "ad",
  admin_label: new TranslatableMarkup("Advertisement slot"),
  category: new TranslatableMarkup("Advertisement"),
)]
class AdSlot extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The ad placement factory.
   *
   * @var \Drupal\ad\Entity\AdPlacementManager
   */
  protected AdPlacementManager $placementFactory;

  /**
   * The ad bucket factory.
   *
   * @var \Drupal\ad\Bucket\BucketFactoryInterface
   */
  protected BucketFactoryInterface $bucketFactory;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    AdPlacementManager $placement_factory,
    BucketFactoryInterface $bucket_factory,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->placementFactory = $placement_factory;
    $this->bucketFactory = $bucket_factory;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('ad.placement_manager'),
      $container->get('ad.bucket_factory'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'bucket_ids' => [],
      'placement_id' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['bucket_ids'] = [
      '#type' => 'checkboxes',
      '#title' => new TranslatableMarkup('Bucket(s)'),
      '#options' => $this->bucketFactory->getList(),
      '#default_value' => $this->configuration['bucket_ids'],
    ];

    $form['placement_id'] = [
      '#type' => 'select',
      '#title' => new TranslatableMarkup('Placement'),
      '#options' => $this->placementFactory->getList(),
      '#default_value' => $this->configuration['placement_id'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['bucket_ids'] = Checkboxes::getCheckedCheckboxes($form_state->getValue('bucket_ids'));
    $this->configuration['placement_id'] = $form_state->getValue('placement_id');
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $bucketIds = $this->configuration['bucket_ids'];
    if (empty($bucketIds)) {
      return [];
    }
    $randomBucketId = $bucketIds[rand(0, count($bucketIds) - 1)];
    $bucket = $this->bucketFactory->get($randomBucketId);
    if (empty($bucket)) {
      return [];
    }
    $placementId = $this->configuration['placement_id'];
    if (empty($placementId)) {
      return [];
    }
    if ($this->placementFactory->get($placementId) == NULL) {
      return [];
    }
    $config = $this->configFactory->get('ad.settings');
    if ($bucket->isEmpty($placementId) && $config->get('hide_empty_blocks')) {
      return [];
    }
    // We're using a placeholder and AJAX to deliver ads to allow tracking
    // impressions:
    return $bucket->buildPlaceholder($placementId);
  }

  /**
   * {@inheritDoc}
   */
  protected function blockAccess(AccountInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'view ads');
  }

}
