<?php

namespace Drupal\Tests\ad\Functional;

use Drupal\ad\Entity\AdPlacement;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the AdPlacement entity CRUD operations.
 *
 * @group ad
 */
class AdPlacementEntityTests extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['ad'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to administer ads.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer ads',
      'administer ad settings',
    ]);
  }

  /**
   * Tests creating an ad placement.
   */
  public function testCreateAdPlacement() {
    $this->drupalLogin($this->adminUser);

    // Navigate to the add form.
    $this->drupalGet('/admin/config/content/ad-placement/add');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Add an advertisement placement');

    // Fill in the form.
    $edit = [
      'label' => 'Test Placement',
      'id' => 'test_placement',
      'description' => 'This is a test placement description.',
    ];
    $this->submitForm($edit, 'Save');

    // Verify the placement was created.
    $this->assertSession()->pageTextContains('Created new advertisement placement Test Placement.');

    // Verify we can load the placement.
    $placement = AdPlacement::load('test_placement');
    $this->assertNotNull($placement);
    $this->assertEquals('Test Placement', $placement->label());
    $this->assertEquals('This is a test placement description.', $placement->getDescription());
  }

  /**
   * Tests deleting an ad placement.
   */
  public function testDeleteAdPlacement() {
    $this->drupalLogin($this->adminUser);

    // Create a placement first.
    $placement = AdPlacement::create([
      'id' => 'test_placement',
      'label' => 'Test Placement',
      'description' => 'Test description',
    ]);
    $placement->save();

    // Delete the placement.
    $this->drupalGet('/admin/config/content/ad-placement/test_placement/delete');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Are you sure you want to delete');

    $this->submitForm([], 'Delete');

    // Verify the placement was deleted.
    $placement = AdPlacement::load('test_placement');
    $this->assertNull($placement);
  }

  /**
   * Tests the ad placement list page.
   */
  public function testAdPlacementList() {
    $this->drupalLogin($this->adminUser);

    // Create a few placements.
    $placement1 = AdPlacement::create([
      'id' => 'placement1',
      'label' => 'Placement 1',
      'description' => 'Description 1',
    ]);
    $placement1->save();

    $placement2 = AdPlacement::create([
      'id' => 'placement2',
      'label' => 'Placement 2',
      'description' => 'Description 2',
    ]);
    $placement2->save();

    // View the list page.
    $this->drupalGet('/admin/config/content/ad-placement');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Advertisement placements');
    $this->assertSession()->pageTextContains('Placement 1');
    $this->assertSession()->pageTextContains('Placement 2');
  }

}
