<?php

declare(strict_types=1);

namespace Drupal\adaptive_interact_client\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManager;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Interact client settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    protected EntityFieldManager $entityFieldManager,
  ) {
    $this->setConfigFactory($config_factory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_field.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'adaptive_interact_client_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['adaptive_interact_client.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    // Add a text field for the widget ID.
    $form['server_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Server URL'),
      '#default_value' => $this->config('adaptive_interact_client.settings')->get('server_url') ?? 'https://interact.adaptive.co.uk',
      '#required' => TRUE,
    ];

    // Add a text field for the widget ID.
    $form['widget_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default widget ID'),
      '#default_value' => $this->config('adaptive_interact_client.settings')->get('widget_id'),
      '#required' => TRUE,
    ];

    // Get all image field definitions for the user entity type.
    $field_definitions = $this->entityFieldManager->getFieldDefinitions('user', 'user');
    $field_options = ['' => $this->t('None')];
    foreach ($field_definitions as $field_name => $definition) {
      $field_type = $definition->getType();
      if ($field_type === 'image') {
        $field_options[$field_name] = $definition->getLabel();
      }
    }

    // Add a select field for choosing the avatar field.
    $form['avatar_field'] = [
      '#type' => 'select',
      '#title' => $this->t('User field to use as an avatar'),
      '#options' => $field_options,
      '#default_value' => $this->config('adaptive_interact_client.settings')->get('avatar_field'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('adaptive_interact_client.settings')
      ->set('server_url', $form_state->getValue('server_url'))
      ->set('widget_id', $form_state->getValue('widget_id'))
      ->set('avatar_field', $form_state->getValue('avatar_field'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
