<?php

declare(strict_types=1);

namespace Drupal\adaptive_interact_client\Plugin\Block;

use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManager;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a chat block block.
 */
#[Block(
  id: 'adaptive_interact_client_chat_block',
  admin_label: new TranslatableMarkup('Chat Block'),
  category: new TranslatableMarkup('Custom'),
)]
final class InteractWidgetBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs the plugin instance.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly AccountInterface $accountInterface,
    private readonly EntityTypeManager $entityTypeManager,
    private readonly FileUrlGeneratorInterface $fileUrlGenerator,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('file_url_generator'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {

    $form['widget_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Widget ID'),
      '#default_value' => $this->configuration['widget_id'] ?? '',
    ];

    $form['prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Initial prompt(s)'),
      '#description' => $this->t('Add single or multiple prompts to be displayed when the chat is opened. Multiple prompts should be separated by a new line.'),
      '#default_value' => $this->configuration['prompt'] ?? '',
    ];

    $form['button_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Button type'),
      '#default_value' => $this->configuration['button_type'] ?? '',
      '#options' => [
        'icon' => $this->t('Icon'),
        'text' => $this->t('Text'),
        'icon_text' => $this->t('Icon & text'),
      ],
    ];

    $form['display_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Button type'),
      '#default_value' => $this->configuration['display_type'] ?? 'modal',
      '#options' => [
        'modal' => $this->t('Modal'),
        'inline' => $this->t('Inline'),
      ],
    ];

    $form['button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Button text'),
      '#default_value' => $this->configuration['button_text'] ?? '',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    // Add button type.
    $data = [
      'buttonType' => $this->configuration['button_type'] ?? 'icon',
      'displayType' => $this->configuration['display_type'] ?? 'modal',
    ];

    // Add default prompt if value exist.
    if (!empty($this->configuration['prompt'])) {
      $data['prompt'] = $this->configuration['prompt'];
    }

    // Add button text if it exists.
    if (!empty($this->configuration['button_text'])) {
      $data['buttonText'] = $this->configuration['button_text'];
    }

    // Add avatar URL if it exists.
    if ($avatar_url = _adaptive_interact_client_get_avatar()) {
      $data['avatarUrl'] = $avatar_url;
    }

    $build = [
      '#theme' => 'adaptive_interact_client__widget',
      '#id' => !empty($this->configuration['widget_id']) ?
      $this->configuration['widget_id'] :
      $this->configFactory->get('adaptive_interact_client.settings')->get('widget_id'),
      '#data' => Json::encode($data),
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    $this->configuration['widget_id'] = $form_state->getValue('widget_id');
    $this->configuration['prompt'] = $form_state->getValue('prompt');
    $this->configuration['button_text'] = $form_state->getValue('button_text');
    $this->configuration['button_type'] = $form_state->getValue('button_type');
    $this->configuration['display_type'] = $form_state->getValue('display_type');
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account): AccessResult {
    // @todo Evaluate the access condition here.
    return AccessResult::allowedIf(TRUE);
  }

}
