<?php

namespace Drupal\address_static_map\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\key\Entity\Key;

/**
 * Implements the module settings form.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'address_static_map.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'address_static_map_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('address_static_map.settings');

    foreach (Key::loadMultiple() as $key) {
      $key_options[$key->id()] = $key->label();
    }

    $form['premier'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Google Maps Premier support'),
      '#description' => $this->t('Use Google Maps Premier support (requires a client ID and crypto key instead of an API key.'),
      '#default_value' => $config->get('premier'),
    ];

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API key'),
      '#description' => $this->t('API key for the service'),
      '#default_value' => $config->get('api_key'),
      '#states' => [
        'visible' => [
          ":input[name=premier]" => ['checked' => FALSE],
        ],
        'required' => [
          ":input[name=premier]" => ['checked' => FALSE],
        ],
      ],
    ];

    $form['secret_key_id'] = [
      '#type' => 'select',
      '#title' => $this->t('URL signing secret key'),
      '#description' => $this->t('Secret key needed to add a signature. If the desired key is not listed, <a href=":link">create a new key</a>.', [':link' => Url::fromRoute('entity.key.add_form')->toString()]),
      '#options' => $key_options,
      '#default_value' => $config->get('secret_key_id'),
      '#states' => [
        'visible' => [
          ":input[name=premier]" => ['checked' => FALSE],
        ],
        'required' => [
          ":input[name=premier]" => ['checked' => FALSE],
        ],
      ],
    ];

    $premium_states = [
      'visible' => [
        ":input[name=premier]" => ['checked' => TRUE],
      ],
      'required' => [
        ":input[name=premier]" => ['checked' => TRUE],
      ],
    ];
    $form['premium_client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Google Premium Client ID'),
      '#description' => $this->t('Client ID for Google Premium'),
      '#default_value' => $config->get('premium_client_id'),
      '#states' => $premium_states,
    ];

    $form['premium_crypto_key_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Google Premium Crypto Key'),
      '#description' => $this->t('Crypto Key for Google Premium. If the desired key is not listed, <a href=":link">create a new key</a>.', [':link' => Url::fromRoute('entity.key.add_form')->toString()]),
      '#options' => $key_options,
      '#default_value' => $config->get('premium_crypto_key_id'),
      '#states' => $premium_states,
    ];

    $form['icon_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom marker icon URL'),
      '#description' => $this->t('Optional URL for custom marker icon to use instead of the regular Google map marker icon. Must be smaller than 64x64.'),
      '#default_value' => $config->get('icon_url'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $config = $this->config('address_static_map.settings');
    if (!$values['premier']) {
      $config
        ->set('api_key', $values['api_key'])
        ->set('secret_key_id', $values['secret_key_id']);
    }
    else {
      $config
        ->set('premium_client_id', $values['premium_client_id'])
        ->set('premium_crypto_key_id', $values['premium_crypto_key_id']);
    }

    $config
      ->set('premier', $values['premier'])
      ->set('icon_url', $values['icon_url'])
      ->save();

    parent::submitForm($form, $form_state);
  }

}
