<?php

namespace Drupal\address_static_map\Service;

use Drupal\Core\Url;

/**
 * Defines a class to render the static Google map.
 */
class MapRenderer {

  /**
   * The MapSigner service.
   *
   * @var \Drupal\address_static_map\Service\MapSigner
   */
  protected MapSigner $mapSigner;

  /**
   * Constructs a new MapRenderer object.
   *
   * @param \Drupal\address_static_map\Service\MapSigner $mapSigner
   *   The MapSigner service.
   */
  public function __construct(MapSigner $mapSigner) {
    $this->mapSigner = $mapSigner;
  }

  /**
   * Render static Google Map image for a specific address.
   *
   * @param string $address
   *   The address being displayed.
   * @param array $settings
   *   An array of settings related to the map to be displayed.
   *
   * @return array
   *   Renderable array to render a Google map image.
   */
  public function renderGoogleMapsImage(string $address, array $settings): array {
    $url_args = [
      'query' => [
        'center' => $address,
        'zoom' => $settings['zoom_level'],
        'size' => $settings['map_size'],
        'scale' => $settings['scale'],
        'maptype' => $settings['map_style'],
        'markers' => implode('|',
          [
            $settings['icon_url'],
            $address,
          ]
        ),
      ],
    ];

    if ($url_args['query']['zoom'] == 'auto') {
      unset($url_args['query']['zoom']);
    }

    // Check for Google Maps API key vs Premium Plan via Client ID.
    if (isset($settings['premier']) && $settings['premier']) {
      $url_args['query']['client'] = $settings['client_id'];
    }
    else {
      $url_args['query']['key'] = $settings['api_key'];
    }

    $settings['staticmap_url'] = Url::fromUri('https://maps.googleapis.com/maps/api/staticmap', $url_args)->toString();

    if (!empty($settings['additional'])) {
      $settings['staticmap_url'] .= '&' . $settings['additional'];
    }

    // Get signature from premium crypto key or APY key signing secret.
    $url = str_replace('https://maps.googleapis.com', '', $settings['staticmap_url']);
    if (isset($settings['premier']) && $settings['premier']) {
      $signature = $this->mapSigner->generateSignature($settings['crypto_key_id'], $url);
    }
    else {
      $signature = $this->mapSigner->generateSignature($settings['secret_key_id'], $url);
    }
    $settings['staticmap_url'] .= '&signature=' . $signature;

    return [
      '#theme' => 'image',
      '#uri' => $settings['staticmap_url'],
      '#alt' => $address,
    ];
  }

}
