<?php

namespace Drupal\admin_feedback\Form;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Database\Database;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for deleting all feedback for a node.
 *
 * @internal
 */
class AdminFeedbackDeleteAllNodeForm extends ConfirmFormBase {

  /**
   * The ID of the feedback to be deleted.
   *
   * @var int
   */
  protected $nodeId;

    /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new VariantPluginFormBase.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  final public function __construct(EntityTypeManagerInterface $entityTypeManager) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static($container->get('entity_type.manager'));
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    $node = $this->entityTypeManager->getStorage('node')->load($this->nodeId);
    $title = $node->getTitle();
    return $this->t('Are you sure you want to delete all feedback for page ":title"?', [':title' => $title]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return Url::fromUri('internal:/node/' . $this->nodeId . '/feedback');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'admin_feedback_delete_all_node_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $id = NULL) {
    $this->nodeId = $id;
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $connection = Database::getConnection();

    // Delete all feedback for this node.
    $query_feedback = $connection->delete('admin_feedback')
      ->condition('nid', $this->nodeId, "=")
      ->execute();

    // Delete the score for this node.
    $query_score = $connection->delete('admin_feedback_score')
      ->condition('nid', $this->nodeId, "=")
      ->execute();

    Cache::invalidateTags(['feedback_cache_tags']);

    // Confirm and redirect to node.
    $this->messenger()->addStatus($this->t('All feedback entries deleted.'));
    $node = $this->entityTypeManager->getStorage('node')->load($this->nodeId);
    $form_state->setRedirectUrl($node->toUrl());
  }

}
