<?php

namespace Drupal\adroll;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Session\AccountInterface;

class Adroll {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The route admin context to determine whether a route is an admin one.
   *
   * @var \Drupal\Core\Routing\AdminContext
   */
  protected $adminContext;

  /**
   * Create an Adroll object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory, AccountInterface $current_user, AdminContext $admin_context) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->adminContext = $admin_context;
  }

  /**
   * Decides whether to track the user or not.
   *
   * @return bool
   *   TRUE if the user should be tracked.
   */
  function shouldTrack(): bool {
    $current_user_roles = $this->currentUser->getRoles();
    $config = $this->configFactory->get('adroll.settings');

    // Check if AdRoll is configured.
    if (!$config->get('adroll_adv_id')) {
      return FALSE;
    }

    // Check if this is an admin page.
    if ($this->adminContext->isAdminRoute()) {
      return FALSE;
    }

    // Check user role segmentation.
    $allowed_roles = array_filter($config->get('adroll_target_roles'), fn($n) => $n);

    // If no checkboxes are selected, allow all roles.
    if (empty($allowed_roles)) {
      return TRUE;
    }

    // Determine if user has required role.
    return (bool) !empty(array_intersect($allowed_roles, $current_user_roles));
  }
}
