<?php

namespace Drupal\adva\Form;

use Drupal\adva\AccessStorage;
use Drupal\adva\Batch\ConsumerAccessRebuildBatchInterface;

use Drupal\adva\Plugin\adva\OverridingAccessConsumerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provide page and form to submit rebuild job.
 */
class RebuildPermissionsForm extends ConfirmFormBase {

  /**
   * Access Consumer being updated.
   *
   * @var \Drupal\adva\Plugin\adva\OverridingAccessConsumerInterface
   */
  protected $consumer;

  /**
   * Access Consumer being updated.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Current Access Storage instance.
   *
   * @var \Drupal\adva\AccessStorage
   */
  protected $accessStorage;

  /**
   * The entity type definition.
   *
   * @var \Drupal\Core\Entity\EntityTypeInterface
   */
  protected $entityType;

  protected ConsumerAccessRebuildBatchInterface $consumerAccessRebuildBatch;

  /**
   * Create an instance of the Form.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_manager
   *   Current Entity Type manager.
   * @param \Drupal\adva\AccessStorage $access_storage
   *   Current Access Storage instance.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_manager,
    AccessStorage $access_storage,
    ConsumerAccessRebuildBatchInterface $consumer_access_rebuild_batch
  ) {
    $this->entityTypeManager = $entity_manager;
    $this->accessStorage = $access_storage;
    $this->consumerAccessRebuildBatch = $consumer_access_rebuild_batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('adva.access_storage'),
      $container->get('adva.batch.consumer_access_rebuild')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'adva_access_rebuild';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to rebuild the permissions for %entityType entities?', [
      '%entityType' => $this->entityType->getLabel(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('system.status');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?OverridingAccessConsumerInterface $consumer = NULL) {
    $this->consumer = $consumer;
    $this->entityType = $this->entityTypeManager->getDefinition($consumer->getEntityTypeId());
    $form = parent::buildForm($form, $form_state);

    if ($consumer->rebuildRequired()) {
      // Unset the primary status of the submit button.
      unset($form['actions']['submit']['#button_type']);

      // Add a new primary batch process button.
      $form['actions']['batch'] = [
        '#type' => 'submit',
        '#value' => $this->t('Process Queued Access Records'),
        '#submit' => [[$this, 'batch']],
        '#weight' => -1,
        '#button_type' => 'primary',
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Rebuild Permissions');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This action rebuilds all permissions on %entityType entites, and may be a lengthy process. Rebuilding will remove all privileges to content and replace them with permissions based on the current modules and settings. This can result in unexpected access to content and this action cannot be undone.', [
      '%entityType' => $this->entityType->getLabel(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->consumer->queue();
    $this->consumerAccessRebuildBatch->createBatch($this->consumer);
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

  /**
   * Batches all queued access records for the entity type.
   *
   * @param array $form
   *   The saved form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state of the saved form.
   */
  public function batch(array &$form, FormStateInterface $form_state) {
    // Batch process all queued access records for the entity type.
    $this->consumerAccessRebuildBatch->createBatch($this->consumer);
  }

}
