<?php

namespace Drupal\adva\Plugin\QueueWorker;

use Drupal\adva\AccessStorageInterface;
use Drupal\adva\Plugin\adva\Manager\AccessConsumerManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a QueueWorker to rebuild access records for every entity type.
 *
 * @QueueWorker(
 *   id = "adva_rebuild_access_records",
 *   title = @Translation("Advanced Access Rebuild Access Records"),
 *   cron = {"time" = 15},
 *   deriver = "Drupal\adva\Plugin\Derivative\RebuildAccessRecordsDeriver"
 * )
 */
class RebuildAccessRecordsQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  public const BASE_QUEUE_ID = 'adva_rebuild_access_records';

  /**
   * The Entity Type Manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The Access Storage service.
   *
   * @var \Drupal\adva\AccessStorageInterface
   */
  protected $accessStorage;

  /**
   * The Consumer Manager service.
   *
   * @var \Drupal\adva\Plugin\adva\Manager\AccessConsumerManagerInterface
   */
  protected $consumerManager;

  /**
   * Creates an instance of the plugin.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The Entity Type Manager service.
   * @param \Drupal\adva\AccessStorageInterface $access_storage
   *   The Access Storage service.
   * @param \Drupal\adva\Plugin\adva\Manager\AccessConsumerManagerInterface $consumer_manager
   *   The Consumer Manager service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, AccessStorageInterface $access_storage, AccessConsumerManagerInterface $consumer_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->entityTypeManager = $entity_type_manager;
    $this->accessStorage = $access_storage;
    $this->consumerManager = $consumer_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('adva.access_storage'),
      $container->get('plugin.manager.adva.consumer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    // Get the consumer for the entity type.
    $consumer = $this
      ->consumerManager
      ->getConsumerForEntityTypeId($data->entity_type_id);

    // Get the storage for then entity type.
    $storage = $this
      ->entityTypeManager
      ->getStorage($data->entity_type_id);

    // Load the entity.
    $entity = $storage->load($data->entity_id);

    // If the entity can't be loaded, exit.
    if (!$entity) {
      return;
    }

    // Process the entity.
    $this->accessStorage->updateRecordsFor($consumer, $entity);

    // Clear the entity's cache.
    $storage->resetCache([$data->entity_id]);
  }

}
