<?php

namespace Drupal\advanced_403_redirect\Plugin\Validation\Constraint;

use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Checks if source URL is unpublished and destination URL is published.
 */
class SourceValidationConstraintValidator extends ConstraintValidator {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The database connection service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs the validator.
   */
  public function __construct() {
    $this->entityTypeManager = \Drupal::service('entity_type.manager');
    $this->database = \Drupal::database();
  }

  /**
   * Validates the constraint.
   *
   * @param mixed $value
   *   The value to validate.
   * @param \Symfony\Component\Validator\Constraint $constraint
   *   The constraint for the validation.
   */
  public function validate($value, Constraint $constraint) {
    $entity = $value->getEntity();
    $source_url = $entity->label->value;
    if (!str_starts_with($source_url, '/')) {
      $this->context->addViolation(
            $constraint->invalidUrlSyntax,
            ['@url' => $source_url]
        );
      return;
    }
    $existing_record = $this->getSourceUrlRecord($source_url);
    if ($existing_record) {
      $this->context->addViolation(
            $constraint->urlAlreadyExist,
            ['@url' => $source_url]
        );
      return;
    }
    $router = \Drupal::service('router');
    try {
      $route_match = $router->match($source_url);
      if ($route_match) {
        $source_url = \Drupal::service('path_alias.manager')
          ->getPathByAlias($source_url);
        $source_entity = $this->getEntityFromPath($source_url);
        if (!$source_entity) {
          $this->context->addViolation(
                $constraint->invalidEntityMessage,
                ['@url' => $source_url]
            );
          return;
        }
        if ($source_entity->isPublished()) {
          $this->context->addViolation(
                $constraint->sourceUnpublishedMessage,
                ['@url' => $source_url]
            );
        }
      }
    }
    catch (\Exception $e) {
      $this->context->addViolation(
            $constraint->invalidRouteMessage,
            ['@url' => $source_url]
        );
      return;
    }

  }

  /**
   * Retrieves an entity from a given path.
   *
   * @param string $path
   *   The internal path to extract the entity from.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The loaded entity or null if not found.
   */
  private function getEntityFromPath($path) {
    if (preg_match('#^/([^/]+)/(\d+)$#', $path, $matches)) {
      $entity_type = $matches[1];
      $entity_id = $matches[2];
      if ($this->entityTypeManager->hasDefinition($entity_type)) {
        return $this->entityTypeManager->getStorage($entity_type)
          ->load($entity_id);
      }
    }
    return NULL;

  }

  /**
   * Retrieves a row from the database if the source URL exists.
   *
   * @param string $source_url
   *   The source URL to check.
   *
   * @return array|null
   *   The database row if found, otherwise NULL.
   */
  private function getSourceUrlRecord($source_url) {
    return $this->database->select('access_denied_url', 'adu')
      ->fields('adu')
      ->condition('label', $source_url)
      ->execute()
      ->fetchAssoc();
  }

}
