<?php

namespace Drupal\advanced_403_redirect\EventSubscriber;

use Drupal\Core\Database\Connection;
use Drupal\Core\EventSubscriber\HttpExceptionSubscriberBase;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;

/**
 * Redirect event subscriber.
 */
class AccessDeniedRedirectSubscriber extends HttpExceptionSubscriberBase {

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The current user (account proxy).
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs event subscriber.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user account proxy.
   * @param \Drupal\Core\Database\Connection $database
   *   The active database connection.
   */
  public function __construct(MessengerInterface $messenger, AccountProxyInterface $current_user, Connection $database) {
    $this->messenger = $messenger;
    $this->currentUser = $current_user;
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  protected function getHandledFormats() {
    return ['html'];
  }

  /**
   * Redirects on 403 Access Denied kernel exceptions.
   *
   * @param \Symfony\Component\HttpKernel\Event\ExceptionEvent $event
   *   The Event to process.
   */
  public function on403(ExceptionEvent $event) : void {
    // Use the injected account proxy instead of \Drupal::currentUser().
    if (!in_array('administrator', $this->currentUser->getRoles())) {
      $request = $event->getRequest();
      $current_path = $request->getPathInfo();

      // Use the injected database connection instead of \Drupal::database().
      $query = $this->database->select('access_denied_url', 'adu');
      $query->fields('adu', ['description', 'message']);
      $query->condition('label', $current_path);
      $result = $query->execute()->fetchAssoc();

      if ($result) {
        $destination_url = $result['description'];
        $status_message = $result['message'];

        if (!empty($status_message)) {
          // second argument can be 'status', 'warning', or 'error'
          $this->messenger->addMessage($status_message, 'warning');
        }

        // Perform the redirection.
        $response = new RedirectResponse($destination_url);
        $event->setResponse($response);
      }
    }
  }

}
