<?php

namespace Drupal\advanced_403_redirect\Plugin\Validation\Constraint;

use Drupal\Core\Routing\RouterInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Validates the destination URL field.
 */
class DestinationValidationConstraintValidator extends ConstraintValidator {

  /**
   * The router service.
   *
   * @var \Drupal\Core\Routing\RouterInterface
   */
  protected $router;

  /**
   * Constructs a destination URL validator.
   *
   * @param \Drupal\Core\Routing\RouterInterface $router
   *   The router service.
   */
  public function __construct(RouterInterface $router) {
    $this->router = $router;
  }

  /**
   * {@inheritdoc}
   */
  public function validate($value, Constraint $constraint): void {
    if (!$constraint instanceof DestinationValidationConstraint) {
      throw new UnexpectedTypeException($constraint, DestinationValidationConstraint::class);
    }

    if ($value === NULL || $value->isEmpty()) {
      return;
    }

    $entity = $value->getEntity();
    $destination_url = $entity->description->value;

    // Ensure URL begins with a slash.
    if (!str_starts_with($destination_url, '/')) {
      $this->context->addViolation(
        $constraint->invalidUrlSyntax,
        ['@url' => $destination_url]
      );
      return;
    }

    try {
      // Use injected router.
      $route_match = $this->router->match($destination_url);
      if ($route_match) {
        return;
      }
    }
    catch (\Exception $e) {
      $this->context->addViolation(
        $constraint->invalidRouteMessage,
        ['@url' => $destination_url]
      );
    }
  }

}
