<?php

namespace Drupal\advanced_403_redirect\Plugin\Validation\Constraint;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Path\AliasManagerInterface;
use Drupal\Core\Routing\RouterInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Validates the source URL field.
 */
class SourceValidationConstraintValidator extends ConstraintValidator {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The router service.
   *
   * @var \Drupal\Core\Routing\RouterInterface
   */
  protected RouterInterface $router;

  /**
   * The alias manager service.
   *
   * @var \Drupal\Core\Path\AliasManagerInterface
   */
  protected AliasManagerInterface $aliasManager;

  /**
   * Constructs a SourceValidationConstraintValidator.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Routing\RouterInterface $router
   *   The router service.
   * @param \Drupal\Core\Path\AliasManagerInterface $alias_manager
   *   The path alias manager service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, Connection $database, RouterInterface $router, AliasManagerInterface $alias_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
    $this->router = $router;
    $this->aliasManager = $alias_manager;
  }

  /**
   * {@inheritdoc}
   */
  public function validate($value, Constraint $constraint): void {
    if (!$constraint instanceof SourceValidationConstraint) {
      throw new UnexpectedTypeException($constraint, SourceValidationConstraint::class);
    }

    if ($value === NULL || $value->isEmpty()) {
      return;
    }

    $entity = $value->getEntity();
    $source_url = (string) $entity->label->value;

    // URL must start with a slash.
    if (!str_starts_with($source_url, '/')) {
      $this->context->addViolation(
            $constraint->invalidUrlSyntax,
            ['@url' => $source_url]
        );
      return;
    }

    // Check if the source URL already exists in our table.
    $existing_record = $this->getSourceUrlRecord($source_url);
    if ($existing_record) {
      $this->context->addViolation(
            $constraint->urlAlreadyExist,
            ['@url' => $source_url]
        );
      return;
    }

    try {
      // Verify route exists for the path.
      $route_match = $this->router->match($source_url);
      if ($route_match) {
        // Convert alias to system path and try to load the entity.
        $internal_path = $this->aliasManager->getPathByAlias($source_url);
        $source_entity = $this->getEntityFromPath($internal_path);
        if ($source_entity === NULL) {
          $this->context->addViolation($constraint->invalidEntityMessage, ['@url' => $source_url]);
          return;
        }
        // If the source entity **is published**, that's a problem for this rule.
        if ($source_entity->isPublished()) {
          $this->context->addViolation(
                $constraint->sourceUnpublishedMessage,
                ['@url' => $source_url]
            );
        }
      }
    }
    catch (\Exception $e) {
      // If the router throws, treat as invalid route.
      $this->context->addViolation($constraint->invalidRouteMessage, ['@url' => $source_url]);
      return;
    }
  }

  /**
   * Load an entity from a system path like /node/123.
   *
   * @param string $path
   *   The internal path (system path) to extract the entity from.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The loaded entity or null.
   */
  private function getEntityFromPath(string $path): ?EntityInterface {
    if (preg_match('#^/([^/]+)/(\d+)$#', $path, $matches)) {
      $entity_type = $matches[1];
      $entity_id = $matches[2];
      if ($this->entityTypeManager->hasDefinition($entity_type)) {
        return $this->entityTypeManager->getStorage($entity_type)
          ->load($entity_id);
      }
    }
    return NULL;
  }

  /**
   * Check for an existing source URL row in the DB.
   *
   * @param string $source_url
   *   The source URL to check.
   *
   * @return array|null
   *   The DB row, or NULL if not found.
   */
  private function getSourceUrlRecord(string $source_url): ?array {
    $row = $this->database->select('access_denied_url', 'adu')
      ->fields('adu')
      ->condition('label', $source_url)
      ->execute()
      ->fetchAssoc();
    return $row === FALSE ? NULL : $row;
  }

}
