/**
 * @file
 * Admin JavaScript for Advanced Country Field.
 */

(function ($, Drupal, debounce) {
  'use strict';

  /**
   * Client-side filtering for country list (like modules page).
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.countryFilterText = {
    attach: function (context, settings) {
      const input = once('country-filter-text', 'input.country-filter-text', context);
      if (!input.length) {
        return;
      }

      const $countries = $('.country-item');
      let searching = false;

      function filterCountryList(e) {
        const query = $(e.target).val().toLowerCase();
        
        // SECURITY FIX: Use simple string matching instead of RegExp
        // to prevent ReDoS (Regular Expression Denial of Service) attacks.
        // This avoids exponential backtracking with malicious patterns like (a+)+b

        function showCountryRow(index, item) {
          const $item = $(item);
          // The wrapper (country-item) has the table-filter-text-source class, so get text directly.
          const text = $item.text().toLowerCase();
          // Use simple includes() instead of regex - safe and fast.
          const textMatch = text.includes(query);
          $item.toggle(textMatch);
        }

        // Filter if the length of the query is at least 2 characters.
        if (query.length >= 2) {
          searching = true;
          $countries.each(showCountryRow);
        } else if (searching) {
          searching = false;
          $countries.show();
        }
      }

      function preventEnterKey(event) {
        if (event.which === 13) {
          event.preventDefault();
          event.stopPropagation();
        }
      }

      $(input[0]).on({
        input: debounce(filterCountryList, 200),
        keydown: preventEnterKey,
      });
    }
  };

  /**
   * Select/Deselect all checkboxes behavior.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.countrySelectAllButtons = {
    attach: function (context, settings) {
      const $selectAllBtn = once('country-select-all', '.country-select-all-btn', context);
      const $deselectAllBtn = once('country-deselect-all', '.country-deselect-all-btn', context);
      
      if ($selectAllBtn.length) {
        $($selectAllBtn[0]).on('click', function(e) {
          e.preventDefault();
          $('input.country-checkbox').prop('checked', true);
          return false;
        });
      }
      
      if ($deselectAllBtn.length) {
        $($deselectAllBtn[0]).on('click', function(e) {
          e.preventDefault();
          $('input.country-checkbox').prop('checked', false);
          return false;
        });
      }
    }
  };

})(jQuery, Drupal, Drupal.debounce);

