<?php

namespace Drupal\advanced_header_field\Plugin\Field\FieldFormatter;

use Drupal\advanced_header_field\AdvancedHeaderFieldInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\link\Plugin\Field\FieldFormatter\LinkFormatter;

/**
 * Plugin implementation of the 'advanced_header_field' html formatter.
 *
 * @FieldFormatter(
 *   id = "advanced_header_field_html",
 *   label = @Translation("HTML Markup"),
 *   field_types = {
 *     "advanced_header_field"
 *   }
 * )
 */
class AdvancedHeaderFieldHtmlFormatter extends LinkFormatter {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'tag' => 'header',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];

    $elements['tag'] = [
      '#type' => 'select',
      '#title' => $this->t('HTML Tag'),
      '#options' => [
        'header' => $this->t('Semantic (header)'),
        'div' => $this->t('Generic (div)'),
      ],
      '#default_value' => $this->getSetting('tag') ?? 'header',
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $settings = $this->getSettings();

    $summary = [];
    $summary[] = $this->t('Render as @tag', ['@tag' => ucfirst($settings['tag'])]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      $elements[$delta] = $this->viewElement($item);
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  protected function viewElement(FieldItemInterface $item) {
    $values = $item->getValue();

    if (!$values['title']) {
      return;
    }

    $options = $values['options'];

    $parent_id = AdvancedHeaderFieldInterface::getHeaderParentId($item);

    /*
     * **************************************************
     * Set - Heading Text
     */

    // Create a link with the URL and title.
    if ($values['uri'] !== 'route:<nolink>') {
      $title = [
        '#type' => 'link',
        '#title' => $item->title,
        '#url' => $this->buildUrl($item),
        '#attributes' => [],
      ];

      if ($options['new_window']) {
        $title['#attributes']['new_window'] = '_blank';
      }
    }
    // No link, just text.
    else {
      $title = $values['title'];
    }

    /*
     * **************************************************
     * Set - Classes
     */

    $classes = [];

    if ($options['size']) {
      $classes[] = 'size-' . $options['size'];
    }

    if ($options['styles']) {
      foreach ($options['styles'] as $style) {
        $classes[] = 'style-' . $style;
      }
    }

    $element = [
      '#theme' => 'advanced_header_field',
      '#header_id' => !empty($options['custom_anchor_id']) ? $options['custom_anchor_id'] : AdvancedHeaderFieldInterface::createAnchorIdFromText($values['title'], $parent_id),
      '#header_tag' => $this->getSetting('tag'),
      '#heading_tag' => $options['heading_tag'],
      '#heading_text' => $title,
      '#subtitle' => $values['subtitle'],
      '#show_in_jump_menu' => $options['show_in_jump_menu'] ?? FALSE,
      '#short_title' => $options['short_title'],
      '#classes' => $classes,
      '#visually_hidden' => $options['visually_hidden'],
    ];

    return $element;
  }

}
