<?php

namespace Drupal\advanced_header_field\Plugin\Field\FieldFormatter;

use Drupal\advanced_header_field\AdvancedHeaderFieldInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'advanced_header_field' summary formatter.
 *
 * @FieldFormatter(
 *   id = "advanced_header_field_summary",
 *   label = @Translation("Summary"),
 *   field_types = {
 *     "advanced_header_field"
 *   }
 * )
 */
class AdvancedHeaderFieldSummaryFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'show_subtitle' => FALSE,
      'show_anchor_id' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];

    $elements['show_subtitle'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Subtitle Field'),
      '#default_value' => $this->getSetting('show_subtitle') ?? FALSE,
    ];

    $elements['show_anchor_id'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Anchor Id'),
      '#default_value' => $this->getSetting('show_anchor_id') ?? FALSE,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      $elements[$delta] = $this->viewElement($item);
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  protected function viewElement(FieldItemInterface $item) {
    $values = $item->getValue();

    if (!$values['title']) {
      return;
    }

    $options = $values['options'];

    $anchor_id = '';
    if ($this->getSetting('show_anchor_id')) {
      $parent_id = AdvancedHeaderFieldInterface::getHeaderParentId($item);

      $anchor_id = !empty($options['custom_anchor_id'])
        ? $options['custom_anchor_id']
        : AdvancedHeaderFieldInterface::createAnchorIdFromText($values['title'], $parent_id);
    }

    $element = [
      '#theme' => 'advanced_header_field_summary',
      '#heading_tag' => strtoupper($options['heading_tag']),
      '#heading_text' => $values['title'],
      '#subtitle' => $this->getSetting('show_subtitle') ? $values['subtitle'] : '',
      '#anchor_id' => $anchor_id,
      '#attached' => [
        'library' => ['advanced_header_field/summary'],
      ],
    ];

    return $element;
  }

}
