<?php

namespace Drupal\advanced_header_field\Plugin\Field\FieldWidget;

use Drupal\advanced_header_field\AdvancedHeaderFieldInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\link\Plugin\Field\FieldWidget\LinkWidget;

/**
 * Plugin implementation of the 'advanced_header_field' widget.
 *
 * @FieldWidget(
 *   id = "advanced_header_field",
 *   label = @Translation("Advanced Header Field"),
 *   field_types = {
 *     "advanced_header_field"
 *   }
 * )
 */
class AdvancedHeaderFieldWidget extends LinkWidget {

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $item = $items[$delta];

    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    $id = uniqid();

    $element['#attributes'] = [
      'class' => ['advanced-header-field'],
      'id' => $id,
    ];
    $element['#type'] = 'details';
    $element['#open'] = TRUE;
    $element['#attached'] = [
      'library' => ['advanced_header_field/admin'],
    ];

    // Rename the title field's title.
    $element['title']['#title'] = 'Heading Text';

    // Make the title required based on the field.
    $element['title']['#required'] = $element['#required'];

    // Add prefix and suffix to the title.
    $element['title']['#prefix'] = '<div class="advanced-header-field__title">';
    $element['title']['#suffix'] = '</div>';

    /*
     * **************************************************
     * Header
     */

    $element['header'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => [
        'class' => ['advanced-header-field__header'],
      ],
    ];

    /*
     * **************************************************
     * Header: Heading
     */

    $element['header']['heading'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => [
        'class' => ['advanced-header-field__heading'],
      ],
    ];

    $semantic_tag_options = array_intersect_key(
      AdvancedHeaderFieldInterface::AVAILABLE_TAGS,
      array_flip($this->getFieldSetting('allowed_tags'))
    );

    $element['header']['heading']['heading_tag'] = [
      '#title' => $this->t('Semantic Tag'),
      '#type' => 'select',
      '#options' => $semantic_tag_options,
      '#default_value' => $item->options['heading_tag'] ?? 'h2',
      '#prefix' => '<div class="advanced-header-field__tag">',
      '#suffix' => '</div>',
    ];

    // Move the 'title' element into the text and unset it.
    $element['header']['heading']['title'] = $element['title'];
    unset($element['title']);

    $element['header']['heading']['subtitle'] = [
      '#title' => $this->t('Subtitle Text'),
      '#type' => 'textfield',
      '#default_value' => $item->subtitle ?? '',
      '#prefix' => '<div class="advanced-header-field__subtitle">',
      '#suffix' => '</div>',
    ];

    /*
     * **************************************************
     * Header: Header Options
     */

    $element['header']['header_options'] = [
      '#type' => 'details',
      '#title' => 'Header Options',
      '#open' => (
        ($item->options['short_title'] ?? '') ||
        ($item->options['show_in_jump_menu'] ?? FALSE)
      ) ?? FALSE,
      '#attributes' => [
        'class' => ['advanced-header-field__header-options'],
      ],
    ];

    $element['header']['header_options']['custom_anchor_id'] = [
      '#title' => $this->t('Custom Anchor Id'),
      '#type' => 'textfield',
      '#default_value' => $item->options['custom_anchor_id'] ?? '',
      '#placeholder' => 'unique-name',
      '#description' => $this->t('The anchor_id is generated by the "Heading text". If you customize the anchor_id, please make sure the value is unique to this page.'),
      '#prefix' => '<div class="advanced-header-field__custom-anchor-name">',
      '#suffix' => '</div>',
    ];

    if (\Drupal::moduleHandler()->moduleExists('advanced_header_field_navigation')) {
      $element['header']['header_options']['show_in_jump_menu'] = [
        '#title' => $this->t('Show in Jump Menu'),
        '#type' => 'checkbox',
        '#default_value' => $item->options['show_in_jump_menu'] ?? FALSE,
        '#prefix' => '<div class="advanced-header-field__jump-menu">',
        '#suffix' => '</div>',
      ];

      $element['header']['header_options']['short_title'] = [
        '#title' => $this->t('Short Title'),
        '#type' => 'textfield',
        '#default_value' => $item->options['short_title'] ?? '',
        '#description' => $this->t('Optional short title used for the "Jump Menu".'),
        '#prefix' => '<div class="advanced-header-field__short-title">',
        '#suffix' => '</div>',
        '#states' => [
          'visible' => [
            "[id='$id'] :input[name*='show_in_jump_menu']" => ['checked' => TRUE],
          ],
        ],
      ];
    }

    /*
     * **************************************************
     * Header: Options Group
     */

    $element['header']['options_group'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => [
        'class' => ['advanced-header-field__options-group'],
      ],
    ];

    /*
     * **************************************************
     * Header: Options Group: Display Options
     */

    $element['header']['options_group']['display_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Display Options'),
      '#open' => (
        ($item->options['visually_hidden'] ?? FALSE) ||
        ($item->options['center'] ?? FALSE) ||
        ($item->options['size'] ?? '') ||
        ($item->options['styles'] ?? [])
      ) ?? FALSE,
      '#attributes' => [
        'class' => [
          'advanced-header-field__options',
          'advanced-header-field__options--display',
        ],
      ],
    ];

    $element['header']['options_group']['display_options']['visually_hidden'] = [
      '#title' => $this->t('Visually hidden'),
      '#type' => 'checkbox',
      '#default_value' => $item->options['visually_hidden'] ?? FALSE,
      '#description' => $this->t('When checked the text will be hidden from display, but not screen readers.'),
    ];

    $element['header']['options_group']['display_options']['size'] = [
      '#title' => $this->t('Size'),
      '#type' => 'select',
      '#options' => AdvancedHeaderFieldInterface::DEFAULT_SIZES,
      '#empty_option' => $this->t('- Select -'),
      '#default_value' => $item->options['size'] ?? '',
    ];

    $element['header']['options_group']['display_options']['styles'] = [
      '#title' => $this->t('Styles'),
      '#type' => 'checkboxes',
      '#options' => $this->getStyleOptions(),
      '#default_value' => $item->options['styles'] ?? [],
    ];

    /*
     * **************************************************
     * Header: Options Group: Link Options
     */

    $element['header']['options_group']['link_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Link Options'),
      '#open' => (
        (!empty($item->uri) && $item->uri !== 'route:<nolink>') ||
        (isset($item->options['new_window']) && $item->options['new_window'])
      ) ?? FALSE,
      '#attributes' => [
        'class' => [
          'advanced-header-field__options',
          'advanced-header-field__options--link',
        ],
      ],
    ];

    // Move the 'uri' element into the link_options and unset it.
    $element['header']['options_group']['link_options']['uri'] = $element['uri'];
    unset($element['uri']);

    $element['header']['options_group']['link_options']['new_window'] = [
      '#title' => $this->t('Open link in new window.'),
      '#type' => 'checkbox',
      '#default_value' => $item->options['new_window'] ?? FALSE,
      '#description' => $this->t("Check this to force link to open in new window."),
    ];

    $element['#element_validate'][] = [get_called_class(), 'processFields'];
    $element['#element_validate'][] = [get_called_class(), 'processOptions'];

    return $element;
  }

  /**
   * Update field values.
   */
  public static function processFields(&$element, FormStateInterface $form_state, $form) {
    $values = $form_state->getValue($element['#parents']);

    $values['uri'] = '';

    if ($element['header']['heading']['title']['#value']) {
      // Change empty uri to '<nolink>'.
      if ($element['header']['options_group']['link_options']['uri']['#value'] === '') {
        $values['uri'] = 'route:<nolink>';
      }
      // Set the uri with the uri's value from the link_options.
      else {
        $values['uri'] = $element['header']['options_group']['link_options']['uri']['#value'];
      }

      // Heading.
      $values['title'] = $element['header']['heading']['title']['#value'];
      $values['subtitle'] = $element['header']['heading']['subtitle']['#value'];
    }

    $form_state->setValueForElement($element, $values);
  }

  /**
   * Copy custom values into the option field.
   */
  public static function processOptions(&$element, FormStateInterface $form_state, $form) {
    $values = $form_state->getValue($element['#parents']);

    // Make sure custom_anchor_id passes the preg_match pattern.
    if (
      $element['header']['header_options']['custom_anchor_id']['#value'] !== '' &&
      !preg_match('/^[a-z][a-z0-9-]+$/', $element['header']['header_options']['custom_anchor_id']['#value'])
    ) {
      $form_state->setError(
        $element['header']['header_options']['custom_anchor_id'],
        t('Custom anchor id must contain only lowercase letters, numbers and hyphens.')
      );
    }

    if ($element['header']['heading']['title']['#value']) {
      $values['options'] = [
        // Heading.
        'heading_tag' => $element['header']['heading']['heading_tag']['#value'],

        // Display Options.
        'visually_hidden' => $element['header']['options_group']['display_options']['visually_hidden']['#value'],
        'size' => $element['header']['options_group']['display_options']['size']['#value'],
        'styles' => isset($element['header']['options_group']['display_options']['styles'])
          ? array_keys($element['header']['options_group']['display_options']['styles']['#value'])
          : [],

        // Link Options.
        'new_window' => $element['header']['options_group']['link_options']['new_window']['#value'],

        // Header Options.
        'custom_anchor_id' => $element['header']['header_options']['custom_anchor_id']['#value'],
      ];

      // Set options if the advanced_header_field_navigation module is enabled.
      if (\Drupal::moduleHandler()->moduleExists('advanced_header_field_navigation')) {
        $values['options']['short_title'] = $element['header']['header_options']['short_title']['#value'];
        $values['options']['show_in_jump_menu'] = $element['header']['header_options']['show_in_jump_menu']['#value'];
      }

      $form_state->setValueForElement($element, $values);
    }

  }

  /**
   * Creates an array of available styles.
   *
   * @return array
   *   An associative array of available styles.
   */
  private function getStyleOptions() {
    $style_options = [];

    // Add the DEFAULT_STYLES to $style_options array.
    $style_options = AdvancedHeaderFieldInterface::DEFAULT_STYLES;

    // Get the custom_styles and apply some function to normalize them.
    $custom_styles = explode("\r\n", $this->getFieldSetting('custom_styles'));
    $custom_styles = array_map('trim', $custom_styles);
    $custom_styles = array_filter($custom_styles, 'strlen');

    foreach ($custom_styles as $custom_style) {
      $matches = [];

      if (preg_match('/(.*)\|(.*)/', $custom_style, $matches)) {
        $key = trim($matches[1]);
        $value = trim($matches[2]);
      }

      $style_options[$key] = $value;
    }

    return $style_options;
  }

}
