<?php

namespace Drupal\advanced_image_media_attributes_formatter\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Security\TrustedCallbackInterface;

// Determinar la clase base dinámicamente solo una vez
if (!class_exists('Drupal\advanced_image_media_attributes_formatter\Plugin\Field\FieldFormatter\MediaWithAttributesFormatterBase', FALSE)) {
  if (class_exists('Drupal\media_image_style_formatter\Plugin\Field\FieldFormatter\RenderedMediaWithImageStyleFormatter')) {
    abstract class MediaWithAttributesFormatterBase extends \Drupal\media_image_style_formatter\Plugin\Field\FieldFormatter\RenderedMediaWithImageStyleFormatter {}
  } else {
    abstract class MediaWithAttributesFormatterBase extends \Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceEntityFormatter {}
  }
}

/**
 * Extended Rendered entity formatter with image loading attributes.
 *
 * @FieldFormatter(
 *   id = "entity_reference_entity_view",
 *   label = @Translation("Rendered entity"),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
class MediaWithAttributesFormatter extends MediaWithAttributesFormatterBase implements TrustedCallbackInterface {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'override_image_loading' => FALSE,
      'image_field_name' => 'field_media_image',
      'image_loading_attribute' => 'lazy',
      'fetchpriority' => '',
      'decoding' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $field_name = $this->fieldDefinition->getName();

    // Detectar si estamos en Views.
    $is_views = $this->isViewsContext($form_state);

    // Obtener allowed bundles (con soporte para Views).
    $allowed_bundles = $this->getAllowedBundles();

    // Get image fields from media bundles
    $image_fields = [];
    $entity_field_manager = \Drupal::service('entity_field.manager');
    $media_bundles = array_keys(\Drupal::entityTypeManager()->getStorage('media_type')->loadMultiple());
    $bundles_to_check = empty($allowed_bundles) ? $media_bundles : array_intersect($allowed_bundles, $media_bundles);

    // Get base fields (excluding thumbnail)
    $base_fields = $entity_field_manager->getBaseFieldDefinitions('media');
    foreach ($base_fields as $base_field_name => $base_definition) {
      if ($base_definition->getType() === 'image' && $base_field_name !== 'thumbnail') {
        $image_fields[$base_field_name] = $base_definition->getLabel() . ' (' . $base_field_name . ') - Base field';
      }
    }

    // Get instance fields by bundle
    foreach ($bundles_to_check as $bundle) {
      $fields = $entity_field_manager->getFieldDefinitions('media', $bundle);
      foreach ($fields as $field_name_bundle => $field_definition) {
        if ($field_definition->getType() === 'image' && !$field_definition->getFieldStorageDefinition()->isBaseField()) {
          $image_fields[$field_name_bundle] = $field_definition->getLabel() . ' (' . $field_name_bundle . ') - Bundle: ' . $bundle;
        }
      }
    }

    // Checkbox principal
    $elements['override_image_loading'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Override image loading'),
      '#default_value' => $this->getSetting('override_image_loading'),
      '#description' => $this->t('Check this option to override the image loading attributes in the media entity.'),
    ];

    if ($is_views) {
      // ===== CONFIGURACIÓN PARA VIEWS (sin #parents, sin wrapper) =====
      $elements['image_field_name'] = [
        '#type' => 'select',
        '#title' => $this->t('Image field to override'),
        '#options' => $image_fields,
        '#default_value' => $this->getSetting('image_field_name'),
        '#description' => $this->t('Select the image type field in the media entity you want to override.'),
        '#empty_value' => '',
        '#states' => [
          'visible' => [
            ':input[name="options[settings][override_image_loading]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $elements['image_loading_attribute'] = [
        '#type' => 'radios',
        '#title' => $this->t('Image loading attribute'),
        '#options' => [
          'lazy' => $this->t('Lazy (<em>loading="lazy"</em>)'),
          'eager' => $this->t('Eager (<em>loading="eager"</em>)'),
        ],
        '#default_value' => $this->getSetting('image_loading_attribute'),
        '#description' => $this->t('Select the loading attribute for images. <a href="@link" target="_blank">Learn more about the loading attribute for images.</a>', [
          '@link' => 'https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-loading',
        ]),
        '#states' => [
          'visible' => [
            ':input[name="options[settings][override_image_loading]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $elements['image_loading_attribute']['lazy']['#description'] = $this->t('Delays loading the image until that section of the page is visible in the browser. When in doubt, lazy loading is recommended.');
      $elements['image_loading_attribute']['eager']['#description'] = $this->t('Force browsers to download an image as soon as possible. This is the browser default for legacy reasons. Only use this option when the image is always expected to render.');

      $elements['fetchpriority'] = [
        '#type' => 'select',
        '#title' => $this->t('Fetchpriority attribute'),
        '#options' => [
          '' => $this->t('- None -'),
          'high' => $this->t('High'),
          'low' => $this->t('Low'),
          'auto' => $this->t('Auto'),
        ],
        '#default_value' => $this->getSetting('fetchpriority'),
        '#description' => $this->t('Hints to the browser how important an image is for the user experience. Use on images that are within the initial viewport.'),
        '#states' => [
          'visible' => [
            ':input[name="options[settings][override_image_loading]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $elements['decoding'] = [
        '#type' => 'select',
        '#title' => $this->t('Decoding attribute'),
        '#options' => [
          '' => $this->t('- None -'),
          'async' => $this->t('Async'),
          'sync' => $this->t('Sync'),
          'auto' => $this->t('Auto'),
        ],
        '#default_value' => $this->getSetting('decoding'),
        '#description' => $this->t('Indicates how the browser should decode the image. "async" for asynchronous decoding (recommended), "sync" for synchronous decoding, "auto" lets the browser decide.'),
        '#states' => [
          'visible' => [
            ':input[name="options[settings][override_image_loading]"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }
    else {
      // ===== CONFIGURACIÓN PARA MANAGE DISPLAY =====
      $elements['image_loading_settings'] = [
        '#type' => 'details',
        '#title' => $this->t('Image loading'),
        '#open' => TRUE,
        '#description' => $this->t('Optimize images with native loading attributes (<em>loading="lazy"</em> or <em>loading="eager"</em>) and fetchpriority hints. This improves performance by controlling when and how browsers load images.'),
        '#states' => [
          'visible' => [
            ':input[name="fields[' . $field_name . '][settings_edit_form][settings][override_image_loading]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $elements['image_loading_settings']['image_field_name'] = [
        '#type' => 'select',
        '#title' => $this->t('Image field to override'),
        '#options' => $image_fields,
        '#default_value' => $this->getSetting('image_field_name'),
        '#description' => $this->t('Select the image type field in the media entity you want to override.'),
        '#empty_value' => '',
        '#parents' => ['fields', $field_name, 'settings_edit_form', 'settings', 'image_field_name'],
        '#weight' => -10,
      ];

      $elements['image_loading_settings']['image_loading_attribute'] = [
        '#type' => 'radios',
        '#title' => $this->t('Image loading attribute'),
        '#options' => [
          'lazy' => $this->t('Lazy (<em>loading="lazy"</em>)'),
          'eager' => $this->t('Eager (<em>loading="eager"</em>)'),
        ],
        '#default_value' => $this->getSetting('image_loading_attribute'),
        '#description' => $this->t('Select the loading attribute for images. <a href="@link" target="_blank">Learn more about the loading attribute for images.</a>', [
          '@link' => 'https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-loading',
        ]),
        '#parents' => ['fields', $field_name, 'settings_edit_form', 'settings', 'image_loading_attribute'],
        '#weight' => 0,
      ];

      $elements['image_loading_settings']['image_loading_attribute']['lazy']['#description'] = $this->t('Delays loading the image until that section of the page is visible in the browser. When in doubt, lazy loading is recommended.');
      $elements['image_loading_settings']['image_loading_attribute']['eager']['#description'] = $this->t('Force browsers to download an image as soon as possible. This is the browser default for legacy reasons. Only use this option when the image is always expected to render.');

      $elements['image_loading_settings']['fetchpriority'] = [
        '#type' => 'select',
        '#title' => $this->t('Fetchpriority attribute'),
        '#options' => [
          '' => $this->t('- None -'),
          'high' => $this->t('High'),
          'low' => $this->t('Low'),
          'auto' => $this->t('Auto'),
        ],
        '#default_value' => $this->getSetting('fetchpriority'),
        '#description' => $this->t('Hints to the browser how important an image is for the user experience. Use on images that are within the initial viewport.'),
        '#parents' => ['fields', $field_name, 'settings_edit_form', 'settings', 'fetchpriority'],
        '#weight' => 10,
      ];

      $elements['image_loading_settings']['decoding'] = [
        '#type' => 'select',
        '#title' => $this->t('Decoding attribute'),
        '#options' => [
          '' => $this->t('- None -'),
          'async' => $this->t('Async'),
          'sync' => $this->t('Sync'),
          'auto' => $this->t('Auto'),
        ],
        '#default_value' => $this->getSetting('decoding'),
        '#description' => $this->t('Indicates how the browser should decode the image. "async" for asynchronous decoding (recommended), "sync" for synchronous decoding, "auto" lets the browser decide.'),
        '#parents' => ['fields', $field_name, 'settings_edit_form', 'settings', 'decoding'],
        '#weight' => 15,
      ];
    }

    return $elements;
  }

  /**
   * Obtiene los bundles permitidos del campo.
   *
   * @return array
   *   Array de bundles permitidos.
   */
  protected function getAllowedBundles() {
    // Método 1: Desde handler_settings (funciona en Manage Display).
    $handler_settings = $this->fieldDefinition->getSetting('handler_settings');
    if (!empty($handler_settings['target_bundles'])) {
      return $handler_settings['target_bundles'];
    }

    // Método 2: Cargar desde el field config original (funciona en Views).
    $field_name = $this->fieldDefinition->getName();
    $entity_type_id = $this->fieldDefinition->getTargetEntityTypeId();

    if ($entity_type_id && $field_name) {
      $field_configs = \Drupal::entityTypeManager()
        ->getStorage('field_config')
        ->loadByProperties([
          'field_name' => $field_name,
          'entity_type' => $entity_type_id,
        ]);

      foreach ($field_configs as $field_config) {
        $settings = $field_config->getSetting('handler_settings');
        if (!empty($settings['target_bundles'])) {
          return $settings['target_bundles'];
        }
      }
    }

    return [];
  }

  /**
   * Detecta si el formulario está siendo usado en contexto de Views.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return bool
   *   TRUE si estamos en Views, FALSE en caso contrario.
   */
  protected function isViewsContext(FormStateInterface $form_state) {
    if ($form_state->has('view')) {
      return TRUE;
    }

    $build_info = $form_state->getBuildInfo();
    $form_id = $build_info['form_id'] ?? '';
    if (strpos($form_id, 'views_ui') !== FALSE) {
      return TRUE;
    }

    $callback = $build_info['callback_object'] ?? NULL;
    if ($callback && strpos(get_class($callback), 'Views') !== FALSE) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $override = $this->getSetting('override_image_loading');

    if ($override) {
      $summary[] = $this->t('Override image loading: Yes');

      $loading_attribute = $this->getSetting('image_loading_attribute');
      $summary[] = $this->t('Loading: @loading', ['@loading' => $loading_attribute]);

      $fetchpriority = $this->getSetting('fetchpriority');
      if (!empty($fetchpriority)) {
        $summary[] = $this->t('Fetchpriority: @priority', ['@priority' => $fetchpriority]);
      }

      $decoding = $this->getSetting('decoding');
      if (!empty($decoding)) {
        $summary[] = $this->t('Decoding: @decoding', ['@decoding' => $decoding]);
      }
    }
    else {
      $summary[] = $this->t('Override image loading: No');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = parent::viewElements($items, $langcode);
    $override = $this->getSetting('override_image_loading');
    $loading_attribute = $this->getSetting('image_loading_attribute');
    $fetchpriority = $this->getSetting('fetchpriority');
    $decoding = $this->getSetting('decoding');

    // Get image_field_name from parent if exists
    $image_field_name = method_exists($this, 'getSetting') ? ($this->getSetting('image_field_name') ?? '') : '';

    if ($override && $image_field_name) {
      foreach ($elements as $delta => &$build) {
        $build['#loading_attribute'] = $loading_attribute;
        $build['#fetchpriority'] = $fetchpriority;
        $build['#decoding'] = $decoding;
        $build['#image_field_name'] = $image_field_name;
        $build['#pre_render'][] = [__CLASS__, 'preRenderAddAttributes'];
      }
    }

    return $elements;
  }

  /**
   * Pre-render callback to add attributes to media images.
   */
  public static function preRenderAddAttributes(array $element) {
    $loading_attribute = $element['#loading_attribute'] ?? '';
    $fetchpriority = $element['#fetchpriority'] ?? '';
    $decoding = $element['#decoding'] ?? '';
    $image_field_name = $element['#image_field_name'] ?? '';

    if (!empty($image_field_name) && isset($element[$image_field_name])) {
      foreach ($element[$image_field_name] as $delta => &$image_element) {
        if (is_numeric($delta)) {
          // For image_formatter theme (uses #item_attributes)
          if (isset($image_element['#theme']) && $image_element['#theme'] === 'image_formatter') {
            if (!isset($image_element['#item_attributes'])) {
              $image_element['#item_attributes'] = [];
            }
            if (!empty($loading_attribute)) {
              $image_element['#item_attributes']['loading'] = $loading_attribute;
            }
            if (!empty($fetchpriority)) {
              $image_element['#item_attributes']['fetchpriority'] = $fetchpriority;
            }
            if (!empty($decoding)) {
              $image_element['#item_attributes']['decoding'] = $decoding;
            }
            unset($image_element['#cache']);
          }

          // For image theme (uses #attributes) - when using original image
          if (isset($image_element['#theme']) && $image_element['#theme'] === 'image') {
            if (!isset($image_element['#attributes'])) {
              $image_element['#attributes'] = [];
            }
            if (!empty($loading_attribute)) {
              $image_element['#attributes']['loading'] = $loading_attribute;
            }
            if (!empty($fetchpriority)) {
              $image_element['#attributes']['fetchpriority'] = $fetchpriority;
            }
            if (!empty($decoding)) {
              $image_element['#attributes']['decoding'] = $decoding;
            }
            unset($image_element['#cache']);
          }

          // For image_style theme (uses #attributes)
          if (isset($image_element['#theme']) && $image_element['#theme'] === 'image_style') {
            if (!isset($image_element['#attributes'])) {
              $image_element['#attributes'] = [];
            }
            if (!empty($loading_attribute)) {
              $image_element['#attributes']['loading'] = $loading_attribute;
            }
            if (!empty($fetchpriority)) {
              $image_element['#attributes']['fetchpriority'] = $fetchpriority;
            }
            if (!empty($decoding)) {
              $image_element['#attributes']['decoding'] = $decoding;
            }
            unset($image_element['#cache']);
          }

          // For nested structure [0]['#theme']
          if (isset($image_element[0]['#theme'])) {
            $nested_theme = $image_element[0]['#theme'];

            if ($nested_theme === 'image_style' || $nested_theme === 'image') {
              if (!isset($image_element[0]['#attributes'])) {
                $image_element[0]['#attributes'] = [];
              }
              if (!empty($loading_attribute)) {
                $image_element[0]['#attributes']['loading'] = $loading_attribute;
              }
              if (!empty($fetchpriority)) {
                $image_element[0]['#attributes']['fetchpriority'] = $fetchpriority;
              }
              if (!empty($decoding)) {
                $image_element[0]['#attributes']['decoding'] = $decoding;
              }
              unset($image_element[0]['#cache']);
            }

            if ($nested_theme === 'image_formatter') {
              if (!isset($image_element[0]['#item_attributes'])) {
                $image_element[0]['#item_attributes'] = [];
              }
              if (!empty($loading_attribute)) {
                $image_element[0]['#item_attributes']['loading'] = $loading_attribute;
              }
              if (!empty($fetchpriority)) {
                $image_element[0]['#item_attributes']['fetchpriority'] = $fetchpriority;
              }
              if (!empty($decoding)) {
                $image_element[0]['#item_attributes']['decoding'] = $decoding;
              }
              unset($image_element[0]['#cache']);
            }
          }
        }
      }
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    $parent_callbacks = [];
    if (method_exists(get_parent_class(static::class), 'trustedCallbacks')) {
      $parent_callbacks = parent::trustedCallbacks();
    }
    return array_merge($parent_callbacks, ['preRenderAddAttributes']);
  }

}
