<?php

namespace Drupal\advanced_image_media_attributes_formatter\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\image\Plugin\Field\FieldFormatter\ImageFormatter;
use Drupal\Core\Security\TrustedCallbackInterface;

/**
 * Extended Image formatter with fetchpriority and decoding attributes.
 */
class ImageFieldWithAttributesFormatter extends ImageFormatter implements TrustedCallbackInterface {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'fetchpriority' => '',
      'decoding' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);
    
    // Detectar si estamos en Views.
    $is_views = $this->isViewsContext($form_state);
    
    if ($is_views) {
      // ===== CONFIGURACIÓN PARA VIEWS (sin #parents) =====
      $elements['fetchpriority'] = [
        '#type' => 'select',
        '#title' => $this->t('Fetchpriority attribute'),
        '#options' => [
          '' => $this->t('- None -'),
          'high' => $this->t('High'),
          'low' => $this->t('Low'),
          'auto' => $this->t('Auto'),
        ],
        '#default_value' => $this->getSetting('fetchpriority'),
        '#description' => $this->t('Hints to the browser how important an image is for the user experience. Use on images that are within the initial viewport.'),
      ];
      $elements['decoding'] = [
        '#type' => 'select',
        '#title' => $this->t('Decoding attribute'),
        '#options' => [
          '' => $this->t('- None -'),
          'async' => $this->t('Async'),
          'sync' => $this->t('Sync'),
          'auto' => $this->t('Auto'),
        ],
        '#default_value' => $this->getSetting('decoding'),
        '#description' => $this->t('Indicates how the browser should decode the image. "async" for asynchronous decoding (recommended), "sync" for synchronous decoding, "auto" lets the browser decide.'),
      ];
    }
    else {
      // ===== CONFIGURACIÓN PARA MANAGE DISPLAY =====
      if (isset($elements['image_loading'])) {
        $field_name = $this->fieldDefinition->getName();
        $elements['image_loading']['fetchpriority'] = [
          '#type' => 'select',
          '#title' => $this->t('Fetchpriority attribute'),
          '#options' => [
            '' => $this->t('- None -'),
            'high' => $this->t('High'),
            'low' => $this->t('Low'),
            'auto' => $this->t('Auto'),
          ],
          '#default_value' => $this->getSetting('fetchpriority'),
          '#description' => $this->t('Hints to the browser how important an image is for the user experience. Use on images that are within the initial viewport.'),
          '#parents' => ['fields', $field_name, 'settings_edit_form', 'settings', 'fetchpriority'],
        ];
        $elements['image_loading']['decoding'] = [
          '#type' => 'select',
          '#title' => $this->t('Decoding attribute'),
          '#options' => [
            '' => $this->t('- None -'),
            'async' => $this->t('Async'),
            'sync' => $this->t('Sync'),
            'auto' => $this->t('Auto'),
          ],
          '#default_value' => $this->getSetting('decoding'),
          '#description' => $this->t('Indicates how the browser should decode the image. "async" for asynchronous decoding (recommended), "sync" for synchronous decoding, "auto" lets the browser decide.'),
          '#parents' => ['fields', $field_name, 'settings_edit_form', 'settings', 'decoding'],
        ];
      }
      else {
        $elements['fetchpriority'] = [
          '#type' => 'select',
          '#title' => $this->t('Fetchpriority attribute'),
          '#options' => [
            '' => $this->t('- None -'),
            'high' => $this->t('High'),
            'low' => $this->t('Low'),
            'auto' => $this->t('Auto'),
          ],
          '#default_value' => $this->getSetting('fetchpriority'),
          '#description' => $this->t('Hints to the browser how important an image is for the user experience. Use on images that are within the initial viewport.'),
        ];
        $elements['decoding'] = [
          '#type' => 'select',
          '#title' => $this->t('Decoding attribute'),
          '#options' => [
            '' => $this->t('- None -'),
            'async' => $this->t('Async'),
            'sync' => $this->t('Sync'),
            'auto' => $this->t('Auto'),
          ],
          '#default_value' => $this->getSetting('decoding'),
          '#description' => $this->t('Indicates how the browser should decode the image. "async" for asynchronous decoding (recommended), "sync" for synchronous decoding, "auto" lets the browser decide.'),
        ];
      }
    }
    
    return $elements;
  }

  /**
   * Detecta si el formulario está siendo usado en contexto de Views.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return bool
   *   TRUE si estamos en Views, FALSE en caso contrario.
   */
  protected function isViewsContext(FormStateInterface $form_state) {
    if ($form_state->has('view')) {
      return TRUE;
    }

    $build_info = $form_state->getBuildInfo();
    $form_id = $build_info['form_id'] ?? '';
    if (strpos($form_id, 'views_ui') !== FALSE) {
      return TRUE;
    }

    $callback = $build_info['callback_object'] ?? NULL;
    if ($callback && strpos(get_class($callback), 'Views') !== FALSE) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $fetchpriority = $this->getSetting('fetchpriority');
    $decoding = $this->getSetting('decoding');

    if (!empty($fetchpriority)) {
      $summary[] = $this->t('Fetchpriority: @priority', ['@priority' => $fetchpriority]);
    }

    if (!empty($decoding)) {
      $summary[] = $this->t('Decoding: @decoding', ['@decoding' => $decoding]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = parent::viewElements($items, $langcode);
    $fetchpriority = $this->getSetting('fetchpriority');
    $decoding = $this->getSetting('decoding');

    if (!empty($fetchpriority) || !empty($decoding)) {
      foreach ($elements as $delta => &$element) {
        $element['#fetchpriority'] = $fetchpriority;
        $element['#decoding'] = $decoding;
        $element['#pre_render'][] = [__CLASS__, 'preRenderAddAttributes'];
      }
    }

    return $elements;
  }

  /**
   * Pre-render callback to add fetchpriority and decoding attributes.
   */
  public static function preRenderAddAttributes(array $element) {
    $fetchpriority = $element['#fetchpriority'] ?? '';
    $decoding = $element['#decoding'] ?? '';

    if (!empty($fetchpriority) || !empty($decoding)) {
      if (isset($element['#theme']) && $element['#theme'] === 'image_formatter') {
        if (!isset($element['#item_attributes'])) {
          $element['#item_attributes'] = [];
        }
        if (!empty($fetchpriority)) {
          $element['#item_attributes']['fetchpriority'] = $fetchpriority;
        }
        if (!empty($decoding)) {
          $element['#item_attributes']['decoding'] = $decoding;
        }
      }

      if (isset($element['#theme']) && $element['#theme'] === 'image_style') {
        if (!isset($element['#attributes'])) {
          $element['#attributes'] = [];
        }
        if (!empty($fetchpriority)) {
          $element['#attributes']['fetchpriority'] = $fetchpriority;
        }
        if (!empty($decoding)) {
          $element['#attributes']['decoding'] = $decoding;
        }
      }
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['preRenderAddAttributes'];
  }

}
