(function ($, Drupal, once) {
  /**
   * Helper to get a unique identifier for a details element using the closest form's data-id.
   */
  const getDetailsId = (el) => {
    // Find the closest form and get its 'data-id' attribute
    const form = el.closest('form');
    return form ? form.getAttribute('data-id') || `form-${form.id}` : null;
  };

  /**
   * Saves the current 'open' state of all details elements for each form to sessionStorage.
   */
  const saveStates = () => {
    const states = {};
    document.querySelectorAll('details').forEach((el) => {
      const formId = getDetailsId(el);
      if (formId) {
        // Create a unique key for each form's state and save it
        const formStateKey = `${formId}-megaMenuDetailsState`;
        if (!states[formStateKey]) {
          states[formStateKey] = {};
        }
        const id =
          el.getAttribute('data-id') ||
          el.id ||
          `details-${el.getAttribute('name')}`;
        states[formStateKey][id] = el.open;
      }
    });
    // Save the entire object with states keyed by form ID
    sessionStorage.setItem('megaMenuDetailsStates', JSON.stringify(states));
  };

  /**
   * Restores 'open' state for each form's details elements from sessionStorage.
   */
  const restoreStates = () => {
    const saved = sessionStorage.getItem('megaMenuDetailsStates');
    if (!saved) {
      return;
    }

    const states = JSON.parse(saved);
    document.querySelectorAll('details').forEach((el) => {
      const formId = getDetailsId(el);
      if (formId) {
        const formStateKey = `${formId}-megaMenuDetailsState`;
        if (states[formStateKey]) {
          const id =
            el.getAttribute('data-id') ||
            el.id ||
            `details-${el.getAttribute('name')}`;
          if (states[formStateKey].hasOwnProperty(id)) {
            el.open = states[formStateKey][id];
          }
        }
      }
    });
  };

  Drupal.behaviors.megaMenuStateJS = {
    attach(context) {
      // 1. Restore states immediately when content is attached
      restoreStates();

      // 2. Apply the 'once' logic to handle event listeners
      once('mega-menu-persist', 'details', context).forEach(
        function (detailsElement) {
          // Listen for manual toggles to update the storage
          detailsElement.addEventListener('toggle', function () {
            saveStates();
          });
        },
      );
    },
  };

  Drupal.AjaxCommands.prototype.megaMenuScrollTo = function (
    ajax,
    response,
    status,
  ) {
    // Adding a slight delay ensures the ReplaceCommand has finished
    // and the browser has painted the new elements.
    setTimeout(function () {
      const $element = $(response.selector);

      if ($element.length) {
        // Use the native DOM element
        const target = $element[0];

        // Comprehensive scroll options
        target.scrollIntoView({
          behavior: 'smooth',
          block: 'center',
          inline: 'nearest',
        });

        // Add a temporary highlight so the user knows it worked
        $element.addClass('ajax-new-item-highlight');
        setTimeout(function () {
          $element.removeClass('ajax-new-item-highlight');
        }, 2000);
      } else {
        console.warn(
          'Mega Menu Scroll: Element not found for selector',
          response.selector,
        );
      }
    }, 150); // 150ms is usually the "sweet spot" for AJAX rendering
  };
})(jQuery, Drupal, once);
