<?php

namespace Drupal\advanced_mega_menu\Controller;

use Drupal\advanced_mega_menu\Entity\MegaMenuContent;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for Mega Menu form per menu item.
 */
class AdvancedMegaMenuController extends ControllerBase {

  /**
   * The form builder service.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Menu link helper service.
   *
   * @var \Drupal\advanced_mega_menu\Menu\MenuLinkHelper
   */
  protected $menuLinkHelper;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->formBuilder = $container->get('form_builder');
    $instance->configFactory = $container->get('config.factory');
    $instance->menuLinkHelper = $container->get('advanced_mega_menu.menu_link_helper');
    return $instance;
  }

  /**
   * Load or create the MegaMenuContent entity and render the form.
   *
   * @param string $menu_id
   *   Menu machine name.
   * @param string $menu_link_content
   *   Menu link plugin ID or UUID (without prefix).
   *
   * @return array|\Symfony\Component\HttpFoundation\JsonResponse
   *   Render array or JSON error response.
   */
  public function action(string $menu_id, string $menu_link_content) {
    // Get menu links via service.
    $menu_links = $this->menuLinkHelper->getMenuLinksWithTitles($menu_id);

    // Validate requested menu link.
    $menu_link_keys = array_column($menu_links, 'key');
    if (!in_array($menu_link_content, $menu_link_keys, TRUE)) {
      throw new NotFoundHttpException();
    }

    /** @var \Drupal\system\Entity\Menu|null $menu */
    $menu = $this->entityTypeManager()
      ->getStorage('menu')
      ->load($menu_id);

    if (!$menu) {
      throw new NotFoundHttpException();
    }

    // Build map: key => title.
    $key_title_map = array_column($menu_links, 'title', 'key');

    $menu_label = $menu->label();
    $menu_link_label = $key_title_map[$menu_link_content] ?? NULL;
    $mega_menu_label = $menu_label . ' - ' . $menu_link_label;

    $menu_link_details = [
      'menu_label' => $menu_label,
      'menu_link_label' => $menu_link_label,
      'mega_menu_label' => $mega_menu_label,
    ];

    // Load or create MegaMenuContent entity.
    $storage = $this->entityTypeManager()->getStorage('megamenu_content');
    $entities = $storage->loadByProperties([
      'form_id' => $menu_link_content,
      'menu_id' => $menu_id,
    ]);
    $mega_menu = $entities ? reset($entities) : NULL;

    if (!$mega_menu) {
      $mega_menu = MegaMenuContent::create([
        'form_id' => $menu_link_content,
        'menu_id' => $menu_id,
        'label' => $mega_menu_label,
        'rows' => [],
        'status' => FALSE,
      ]);
    }

    return $this->formBuilder->getForm(
      'Drupal\advanced_mega_menu\Form\MegaMenuLayoutBuilderForm',
      $mega_menu,
      $menu_link_details
    );
  }

  /**
   * Enable or disable Mega Menu for a menu.
   *
   * @param string $menu_id
   *   Menu machine name.
   * @param string $action
   *   'Enable' or 'Disable'.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response.
   */
  public function toggle(string $menu_id, string $action): RedirectResponse {
    $config = $this->configFactory
      ->getEditable('advanced_mega_menu.settings');

    $enabled_menus = $config->get('enabled_menus') ?? [];

    if ($action === 'enable') {
      $enabled_menus[] = $menu_id;
      $enabled_menus = array_unique($enabled_menus);
    }
    else {
      $enabled_menus = array_diff($enabled_menus, [$menu_id]);
    }

    $config
      ->set('enabled_menus', $enabled_menus)
      ->save();

    $this->messenger()->addStatus($this->t(
      'Mega Menu @menu has been @action.',
      [
        '@menu' => $menu_id,
        '@action' => $action === 'enable' ? 'enabled' : 'disabled',
      ]
    ));

    return new RedirectResponse(
      Url::fromRoute('entity.menu.collection')->toString()
    );
  }

}
