<?php

namespace Drupal\advanced_mega_menu\Entity;

use Drupal\advanced_mega_menu\MegaMenuContentInterface;
use Drupal\advanced_mega_menu\Service\MegaMenuBlockRenderer;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Template\Attribute;
use Drupal\Component\Utility\Html;

/**
 * Defines the MegaMenuContent entity.
 *
 * @ConfigEntityType(
 *   id = "megamenu_content",
 *   label = @Translation("Mega Menu Content"),
 *   handlers = {
 *     "list_builder" = "Drupal\advanced_mega_menu\MegaMenuContentListBuilder",
 *     "form" = {
 *       "add" = "Drupal\advanced_mega_menu\Form\MegaMenuContentForm",
 *       "edit" = "Drupal\advanced_mega_menu\Form\MegaMenuContentForm",
 *       "delete" = "Drupal\advanced_mega_menu\Form\MegaMenuContentDeleteForm"
 *     }
 *   },
 *   config_prefix = "megamenu_content",
 *   admin_permission = "administer menus",
 *   list_cache_tags = {"rendered"},
 *   entity_keys = {
 *     "id" = "form_id",
 *     "label" = "label",
 *     "status" = "status"
 *   },
 *   config_export = {
 *     "form_id",
 *     "menu_id",
 *     "label",
 *     "rows",
 *     "display_settings",
 *     "uuid"
 *   },
 *   links = {
 *     "collection" = "/admin/structure/advanced-mega-menu",
 *     "delete-form" = "/admin/structure/advanced-mega-menu/{megamenu_content}/delete"
 *   }
 * )
 */
class MegaMenuContent extends ConfigEntityBase implements MegaMenuContentInterface {

  /**
   * The label.
   *
   * @var string
   */
  protected $label;

  /**
   * The form ID.
   *
   * @var string
   */
  public $form_id;

  /**
   * The menu ID.
   *
   * @var string
   */
  public $menu_id;

  /**
   * Rows data.
   *
   * @var array
   */
  public $rows;

  /**
   * Status.
   *
   * @var bool
   */
  public $status;

  /**
   * Display settings.
   *
   * @var array
   */
  public $display_settings;

  /**
   * Lazy-loaded block renderer service.
   */
  protected ?MegaMenuBlockRenderer $blockRenderer = NULL;

  /**
   * Get block renderer service.
   */
  protected function blockRenderer(): MegaMenuBlockRenderer {
    if (!$this->blockRenderer) {
      $this->blockRenderer = \Drupal::service('advanced_mega_menu.block_renderer');
    }
    return $this->blockRenderer;
  }

  /**
   * Returns the unique ID of this entity.
   *
   * @return string
   *   The form ID of this mega menu entity.
   */
  public function id() {
    return $this->form_id;
  }

  /**
   * Gets the form ID of this mega menu entity.
   *
   * @return string
   *   The form ID.
   */
  public function getFormId() {
    return $this->form_id;
  }

  /**
   * Sets the form ID for this mega menu entity.
   *
   * @param string $form_id
   *   The form ID.
   */
  public function setFormId($form_id) {
    $this->form_id = $form_id;
  }

  /**
   * Gets the menu ID associated with this mega menu entity.
   *
   * @return string
   *   The menu ID.
   */
  public function getMenuId() {
    return $this->menu_id;
  }

  /**
   * Sets the menu ID for this mega menu entity.
   *
   * @param string $menu_id
   *   The menu ID.
   */
  public function setMenuId($menu_id) {
    $this->menu_id = $menu_id;
  }

  /**
   * Gets the human-readable label for this mega menu entity.
   *
   * @return string
   *   The label.
   */
  public function getLabel() {
    return $this->label;
  }

  /**
   * Sets the human-readable label for this mega menu entity.
   *
   * @param string $label
   *   The label.
   */
  public function setLabel($label) {
    $this->label = $label;
  }

  /**
   * Gets the structured rows for the mega menu.
   *
   * @return array
   *   An array of rows, each containing columns and fields.
   */
  public function getRows() {
    return $this->rows ?? [];
  }

  /**
   * Sets the structured rows for the mega menu.
   *
   * @param array $rows
   *   An array of rows, each containing columns and fields.
   */
  public function setRows($rows) {
    $this->set('rows', $rows);
    return $this;
  }

  /**
   * Gets the status of this mega menu entity.
   *
   * @return bool
   *   TRUE if enabled, FALSE otherwise.
   */
  public function getStatus() {
    return $this->status ?? FALSE;
  }

  /**
   * Sets the status of this mega menu entity.
   *
   * @param bool $status
   *   TRUE to enable, FALSE to disable.
   */
  public function setStatus($status) {
    $this->set('status', $status);
    return $this;
  }

  /**
   * Gets the display settings of this mega menu entity.
   *
   * @return array
   *   An associative array of display settings.
   */
  public function getDisplaySettings() {
    $settings = $this->display_settings ?? [];

    if (!empty($settings['wrapper_class'])) {
      $settings['wrapper_class'] = Html::cleanCssIdentifier($settings['wrapper_class']);
    }

    if (!empty($settings['row_class'])) {
      $settings['row_class'] = Html::cleanCssIdentifier($settings['row_class']);
    }

    if (!empty($settings['column_class'])) {
      $settings['column_class'] = Html::cleanCssIdentifier($settings['column_class']);
    }

    return $settings;

  }

  /**
   * Sets the display settings for this mega menu entity.
   *
   * @param array $display_settings
   *   An associative array of display settings.
   */
  public function setDisplaySettings($display_settings) {
    $this->set('display_settings', $display_settings);
    return $this;
  }

  /**
   * Build structured mega menu render arrays.
   */
  public function getRowsData(): array {
    $rows = $this->getRows();

    $result = [];
    $row_index = 0;
    $row_attributes = new Attribute();
    $display_settings = $this->display_settings;

    foreach ($rows as $row) {

      if (empty($row['columns']) || !is_array($row['columns'])) {
        continue;
      }

      $col_index = 0;

      $column_layout_mode = $row['column_layout_mode'] ?? '';
      $single_row_class = $row['single_row_class'] ?? '';

      if ($single_row_class !== '') {
        $single_row_class = Html::cleanCssIdentifier($single_row_class);
      }

      if ($column_layout_mode == 'custom') {
        $row_attributes->addClass([
          $single_row_class,
        ]);
      }
      else {
        $row_class = $display_settings['row_class'] ?? '';
        if ($row_class !== '') {
          $row_class = Html::cleanCssIdentifier($row_class);
        }

        $row_attributes->addClass([
          $column_layout_mode,
          'advanced-mega-menu-row',
          $row_class,
        ]);
      }

      $result[$row_index]['row_attributes'] = $row_attributes;

      $result[$row_index]['column_layout_mode'] = $column_layout_mode;

      foreach ($row['columns'] as $col) {
        if (empty($col['blocks']) || !is_array($col['blocks'])) {
          continue;
        }

        $field_index = 0;

        $column_attributes = new Attribute();
        if ($column_layout_mode == 'custom') {
          $col_class = $col['col_class'] ?? '';

          if ($col_class !== '') {
            $col_class = Html::cleanCssIdentifier($col_class);
          }

          $column_attributes->addClass([
            $col_class,
          ]);
        }
        else {
          $column_class = $display_settings['column_class'] ?? '';
          if ($column_class !== '') {
            $column_class = Html::cleanCssIdentifier($column_class);
          }

          $column_attributes->addClass([
            'advanced-mega-menu-col',
            $column_class,
          ]);
        }
        $result[$row_index]['column_attributes'][$col_index] = $column_attributes;

        foreach ($col['blocks'] as $field) {
          if (empty($field['category'])) {
            continue;
          }

          $category = $field['category'];

          if (empty($field[$category])) {
            continue;
          }

          $show_label = $field['show_label'] ?? FALSE;

          $block_info = $field[$category] ?? [];

          $block_details = [];
          switch ($category) {
            case 'view_id':
              $details = explode(':', $block_info);
              if (isset($details[1])) {
                $block_details['view_id'] = $details[0];
                $block_details['display_id'] = $details[1];
              }

              break;
          }

          $renderer = $this->blockRenderer();

          $result[$row_index]['columns'][$col_index][$field_index]['block'] = match ($category) {
            'theme_id'   => $renderer->renderThemeBlock($block_info, $show_label),
            'view_id'    => $renderer->renderViewBlock($block_details, $show_label),
            'content_id' => $renderer->renderContentBlock($block_info, $show_label),
            default         => [],
          };

          $field_index++;
        }

        $col_index++;
      }

      $row_index++;
    }

    return $result;
  }

}
