<?php

namespace Drupal\advanced_mega_menu\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Drupal\Component\Utility\Xss;

/**
 * Configuration form for Mega Menu Content.
 *
 * Allows site admins to enable Mega Menu functionality on specific menus.
 */
class MegaMenuContentSettingsForm extends ConfigFormBase {

  /**
   * The menu storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $menuStorage;

  /**
   * Constructs a MegaMenuContentSettingsForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->menuStorage = $entity_type_manager->getStorage('menu');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['advanced_mega_menu.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_mega_menu_settings_form';
  }

  /**
   * Builds the Mega Menu settings form.
   *
   * Displays all site menus as checkboxes with optional "Configure Mega
   * Menu" link.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return array
   *   The built form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('advanced_mega_menu.settings');
    $menus = $this->menuStorage->loadMultiple();
    $enabled_menus = $config->get('enabled_menus') ?: [];

    $form['enabled_menus'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Enable Mega Menu for specific menus'),
      '#description' => $this->t('Select the menus that should have Mega Menu functionality. After enabling, you can configure each menu by clicking the "Configure Mega Menu" link.'),
    ];

    // Save menu keys in form state to use in submitForm().
    $menu_keys = array_keys($menus);
    $form_state->set('menu_keys', $menu_keys);

    foreach ($menus as $menu) {
      $menu_id = $menu->id();
      $menu_label = $menu->label();

      $is_enabled = in_array($menu_id, $enabled_menus);

      // Add a "Configure Mega Menu" link if the menu is enabled.
      $link = '';
      if ($is_enabled) {
        $edit_url = Url::fromRoute('entity.menu.edit_form', ['menu' => $menu_id]);
        $link = Link::fromTextAndUrl($this->t('Configure Mega Menu'), $edit_url)->toString();
      }

      // Checkbox with label and inline link.
      $form['enabled_menus'][$menu_id] = [
        '#type' => 'checkbox',
        '#title' => $menu_label,
        '#description' => $link,
        '#default_value' => $is_enabled,
      ];
    }

    // ==========================================================================
    // Section: Submenu icons & accessibility
    // ==========================================================================
    // Load global icon settings from config.
    $icon_settings = $config->get('submenu_icon') ?? [];

    $form['submenu_icons'] = [
      '#type' => 'details',
      '#title' => $this->t('Submenu icons & accessibility'),
      '#description' => $this->t('Configure icons, symbols, and accessibility labels for expandable submenus.'),
      '#open' => TRUE,
    ];

    // Enable submenu icon.
    $form['submenu_icons']['submenu_icon_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show submenu icon'),
      '#description' => $this->t('Displays an indicator icon next to menu items with submenus.'),
      '#default_value' => $icon_settings['enabled'] ?? 0,
    ];

    // Separate expand / collapse icons.
    $form['submenu_icons']['submenu_icon_separate_states'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use separate icons for expand and collapse'),
      '#description' => $this->t('Allows different icons and ARIA labels for collapsed and expanded states.'),
      '#default_value' => $icon_settings['separate_states'] ?? 0,
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Icon source selection (predefined symbols or icon fonts).
    $form['submenu_icons']['submenu_icon_source'] = [
      '#type' => 'radios',
      '#title' => $this->t('Icon source'),
      '#description' => $this->t('Choose whether to use simple symbols or icon fonts / CSS-based icons.'),
      '#options' => [
        'unicode' => $this->t('Predefined symbols'),
        'icon_font' => $this->t('Icon font / CSS class'),
      ],
      '#default_value' => $icon_settings['source'] ?? 'unicode',
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // --------------------------
    // Sub-details: Unicode icons
    // --------------------------
    $form['submenu_icons']['unicode_icons'] = [
      '#type' => 'details',
      '#title' => $this->t('Predefined symbols'),
      '#description' => $this->t('Choose symbols for expand/collapse states.'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_source"]' => ['value' => 'unicode'],
          ':input[name="submenu_icon_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['submenu_icons']['unicode_icons']['submenu_icon_unicode_expand'] = [
      '#type' => 'select',
      '#title' => $this->t('Expand icon symbol'),
      '#options' => [
        '▼' => $this->t('▼ Down'),
        '+' => $this->t('+ Plus'),
      ],
      '#default_value' => $icon_settings['unicode_expand'] ?? '▶',
    ];

    $form['submenu_icons']['unicode_icons']['submenu_icon_unicode_collapse'] = [
      '#type' => 'select',
      '#title' => $this->t('Collapse icon symbol'),
      '#options' => [
        '▼' => $this->t('▼ Down'),
        '▲' => $this->t('▲ Up'),
        '−' => $this->t('− Minus'),
      ],
      '#default_value' => $icon_settings['unicode_collapse'] ?? '▼',
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_separate_states"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // --------------------------
    // Sub-details: Icon fonts / CSS classes
    // --------------------------
    $form['submenu_icons']['icon_fonts'] = [
      '#type' => 'details',
      '#title' => $this->t('Icon fonts / CSS classes'),
      '#description' => $this->t('Use icon fonts like FontAwesome, Flaticon, or your custom CSS classes.'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_source"]' => ['value' => 'icon_font'],
          ':input[name="submenu_icon_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['submenu_icons']['icon_fonts']['submenu_icon_font_expand'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Expand icon class'),
      '#description' => $this->t('Example: fa-solid fa-chevron-right, fi fi-arrow-down'),
      '#default_value' => $icon_settings['font_expand'] ?? '',
    ];

    $form['submenu_icons']['icon_fonts']['submenu_icon_font_collapse'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Collapse icon class'),
      '#description' => $this->t('Example: fa-solid fa-chevron-down, fi fi-arrow-up'),
      '#default_value' => $icon_settings['font_collapse'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_separate_states"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['submenu_icons']['icon_fonts']['submenu_icon_element'] = [
      '#type' => 'select',
      '#title' => $this->t('Icon HTML element'),
      '#options' => [
        'span' => $this->t('span'),
        'i' => $this->t('i'),
      ],
      '#default_value' => $icon_settings['element'] ?? 'span',
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_source"]' => ['value' => 'icon_font'],
        ],
      ],
    ];

    // --------------------------
    // Sub-details: ARIA labels
    // --------------------------
    $form['submenu_icons']['aria_labels'] = [
      '#type' => 'details',
      '#title' => $this->t('Accessibility labels (ARIA)'),
      '#description' => $this->t('Provide screen reader labels for submenu toggle icons.'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['submenu_icons']['aria_labels']['submenu_icon_aria'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ARIA label'),
      '#description' => $this->t('Accessibility label for submenu toggle when same icon is used for expand and collapse.'),
      '#default_value' => $icon_settings['aria'] ?? 'Expand submenu',
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_separate_states"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['submenu_icons']['aria_labels']['submenu_icon_aria_expand'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ARIA label (expand)'),
      '#description' => $this->t('Screen reader label when submenu is collapsed.'),
      '#default_value' => $icon_settings['aria_expand'] ?? 'Expand submenu',
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_separate_states"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['submenu_icons']['aria_labels']['submenu_icon_aria_collapse'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ARIA label (collapse)'),
      '#description' => $this->t('Screen reader label when submenu is expanded.'),
      '#default_value' => $icon_settings['aria_collapse'] ?? 'Collapse submenu',
      '#states' => [
        'visible' => [
          ':input[name="submenu_icon_separate_states"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['disable_assets'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable Mega Menu JavaScript and CSS'),
      '#description' => $this->t('Check this option if you want to use your own custom JavaScript and CSS instead of the assets provided by the Advanced Mega Menu module.'),
      '#default_value' => $config->get('disable_assets') ?? FALSE,
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // 2. Validate Icon Font / CSS Classes.
    if ($form_state->getValue('submenu_icon_source') === 'icon_font') {

      $font_fields = [
        'submenu_icon_font_expand' => $this->t('Expand icon class'),
        'submenu_icon_font_collapse' => $this->t('Collapse icon class'),
      ];

      foreach ($font_fields as $field_key => $label) {
        // Only validate collapse if "separate states" is checked.
        if ($field_key === 'submenu_icon_font_collapse' && !$form_state->getValue('submenu_icon_separate_states')) {
          continue;
        }

        // The HTML name path used to highlight the field in red on error.
        $element_path = "submenu_icons][icon_fonts][$field_key";

        // Validate Expand/Collapse class format.
        $this->validateClassField(
          $form_state,
          $field_key,
          $element_path,
          $this->t('The @field contains invalid characters or starts with a number.', [
            '@field' => $label,
          ])
        );

        // Required field validation for font classes.
        if (empty($form_state->getValue($field_key))) {
          $form_state->setErrorByName($element_path, $this->t('The @field is required when using Icon Fonts.', [
            '@field' => $label,
          ]));
        }
      }
    }

    // 3. Validate ARIA Accessibility Labels.
    $aria_path_prefix = 'submenu_icons][aria_labels][';

    if ($form_state->getValue('submenu_icon_separate_states')) {
      // Validate Expand ARIA.
      if (empty($form_state->getValue('submenu_icon_aria_expand'))) {
        $form_state->setErrorByName($aria_path_prefix . 'submenu_icon_aria_expand', $this->t('The Expand ARIA label is required.'));
      }
      // Validate Collapse ARIA.
      if (empty($form_state->getValue('submenu_icon_aria_collapse'))) {
        $form_state->setErrorByName($aria_path_prefix . 'submenu_icon_aria_collapse', $this->t('The Collapse ARIA label is required.'));
      }
    }
    else {
      // Validate Single ARIA label.
      if (empty($form_state->getValue('submenu_icon_aria'))) {
        $form_state->setErrorByName($aria_path_prefix . 'submenu_icon_aria', $this->t('The ARIA label is required.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('advanced_mega_menu.settings');
    $menu_keys = $form_state->get('menu_keys');

    // 1. Handle Enabled Menus.
    if ($menu_keys) {
      $enabled_menus = [];
      foreach ($menu_keys as $menu_key) {
        if ($form_state->getValue($menu_key)) {
          $enabled_menus[] = $menu_key;
        }
      }
      $config->set('enabled_menus', $enabled_menus);
    }

    // 2. Handle Global Asset Setting.
    $config->set('disable_assets', (bool) $form_state->getValue('disable_assets'));

    // Helper: Sanitize CSS classes.
    // We use a strict regex to allow only valid class characters.
    $sanitize_classes = function ($value) {
      $value = trim($value ?? '');
      // Strip everything except alphanumeric, spaces, hyphens, and underscores.
      return preg_replace('/[^a-zA-Z0-9\s\-_]/', '', $value);
    };

    // Helper: Sanitize ARIA labels using Xss filter.
    // Xss::filter() is good for string-based labels, though Html::escape
    // is usually preferred for attributes.
    $sanitize_label = function ($value) {
      return Xss::filter(trim($value ?? ''));
    };

    // 3. Handle Global Submenu Icon & Accessibility Settings.
    $icon_data = [
      'enabled'         => (int) ($form_state->getValue('submenu_icon_enabled') ?? 0),
      'separate_states' => (int) ($form_state->getValue('submenu_icon_separate_states') ?? 0),
      'source'          => $form_state->getValue('submenu_icon_source') ?? 'unicode',

      // Unicode selections.
      'unicode_expand'   => trim($form_state->getValue('submenu_icon_unicode_expand') ?? '▶'),
      'unicode_collapse' => trim($form_state->getValue('submenu_icon_unicode_collapse') ?? '▼'),

      // Font / CSS classes (Strict sanitization).
      'font_expand'      => $sanitize_classes($form_state->getValue('submenu_icon_font_expand')),
      'font_collapse'    => $sanitize_classes($form_state->getValue('submenu_icon_font_collapse')),
      'element'          => $form_state->getValue('submenu_icon_element') ?? 'span',

      // ARIA labels (Using Xss filter).
      'aria'             => $sanitize_label($form_state->getValue('submenu_icon_aria')),
      'aria_expand'      => $sanitize_label($form_state->getValue('submenu_icon_aria_expand')),
      'aria_collapse'    => $sanitize_label($form_state->getValue('submenu_icon_aria_collapse')),
    ];

    $config->set('submenu_icon', $icon_data);
    $config->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Validates a CSS class field.
   */
  private function validateClassField(
    FormStateInterface $form_state,
    $value_path,
    string $error_name,
    string $message,
  ): void {
    $value = $form_state->getValue($value_path);
    if ($value && !advanced_mega_menu_are_valid_classes($value)) {
      $form_state->setErrorByName($error_name, $message);
    }
  }

}
