<?php

namespace Drupal\advanced_mega_menu\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\advanced_mega_menu\Entity\MegaMenuContent;
use Drupal\views\Views;
use Drupal\Component\Utility\Html;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\advanced_mega_menu\Ajax\ScrollToCommand;
use Psr\Log\LoggerInterface;

/**
 * Full Fixed Mega Menu Form.
 */
class MegaMenuLayoutBuilderForm extends FormBase implements ContainerInjectionInterface {

  /**
   * Configuration name.
   *
   * @var string
   */
  const CONFIG_NAME = 'advanced_mega_menu.settings';

  /**
   * A request stack symfony instance.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Maximum number of rows in the mega menu.
   *
   * Hard safety limit to prevent excessive layout storage.
   *
   * @var int
   */
  private const MAX_ROWS = 30;

  /**
   * Maximum number of columns per row in the mega menu.
   *
   * Hard safety limit to prevent excessive layout storage.
   *
   * @var int
   */
  private const MAX_COLUMNS_PER_ROW = 24;

  /**
   * Maximum number of blocks per column in the mega menu.
   *
   * Hard safety limit to prevent excessive layout storage.
   *
   * @var int
   */
  private const MAX_BLOCKS_PER_COLUMN = 16;

  /**
   * Constructs a new MegaMenuLayoutBuilderForm object.
   *
   * @param \Drupal\Core\Stack\RequestStack $request_stack
   *   The request stack service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(RequestStack $request_stack, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, LoggerInterface $logger) {
    $this->requestStack = $request_stack;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    // Inject the RequestStack service.
    return new static(
      $container->get('request_stack'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory')->get('advanced_mega_menu'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_mega_menu_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['advanced_mega_menu.settings'];
  }

  /**
   * Builds the Advanced Mega Menu settings form.
   *
   * Initializes form state using the provided Mega Menu entity and menu link
   * details, and prepares the configuration form used to manage mega menu
   * layout and behavior.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param \Drupal\advanced_mega_menu\Entity\MegaMenuContent|null $mega_menu
   *   (optional) The Mega Menu content entity being edited.
   * @param array|null $menu_link_details
   *   (optional) An associative array containing menu link metadata such as
   *   menu label and menu link label.
   *
   * @return array
   *   The completed form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?MegaMenuContent $mega_menu = NULL, $menu_link_details = NULL) {
    // --- 1. Form Setup and State Initialization ---
    $menu_label = $menu_link_details['menu_label'] ?? 'None';
    $menu_link_label = $menu_link_details['menu_link_label'] ?? 'None';
    $mega_menu_label = $menu_link_details['mega_menu_label'] ?? 'None';
    $form_state->set('mega_menu_entity', $mega_menu);
    $form_state->set('mega_menu_label', $mega_menu_label);

    // Dynamic, descriptive form title.
    $form_title = $this->t('Build Mega Menu Layout for <em>@link</em> in <em>@menu</em>', [
      '@menu' => $menu_label,
      '@link' => $menu_link_label,
    ]);

    $form['#title'] = $form_title;
    $form['#attributes'] = [
      'data-id' => $mega_menu->id(),
    ];

    $form['#tree'] = TRUE;
    $form['#attached']['library'][] = 'advanced_mega_menu/advanced_mega_menu_form';

    $layout_options = [
      'auto'         => $this->t('Auto – Columns sized based on content'),
      'equal'        => $this->t('Equal – All columns share equal width'),
      'fraction'     => $this->t('Fractional – Columns divide space evenly (1/n)'),
      'wrap'         => $this->t('Flexible Wrap – Columns shrink and wrap automatically'),
      'minmax'       => $this->t('Min/Max Flex – Columns grow/shrink within min/max limits'),
      'proportional' => $this->t('Proportional – Columns grow based on content size'),
      'rfrac'        => $this->t('Flexible Fraction – Columns flexibly resize across container'),
      'dynstack'     => $this->t('Dynamic Stack – Columns automatically stack when space is limited'),
      'fauto'        => $this->t('Flexible Auto – Columns fill space dynamically'),
      'adapt'        => $this->t('Adaptive Flex – Columns grow/shrink but never overflow'),
      'columns-2'     => $this->t('2 Columns – Flex items split into 2 equal columns'),
      'columns-3'     => $this->t('3 Columns – Flex items split into 3 equal columns'),
      'columns-4'     => $this->t('4 Columns – Flex items split into 4 equal columns'),
      'columns-5'     => $this->t('5 Columns – Flex items split into 5 equal columns'),
      'columns-6'     => $this->t('6 Columns – Flex items split into 6 equal columns'),
    ];

    // Populate source options.
    $this->prepareSourceOptions($form_state);

    $display_settings = $mega_menu->getDisplaySettings('menu_display_settings');

    $form['messages'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages-section']],
    ];

    $form['enable_mega_menu'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable mega menu for <em>@link</em> (<em>@menu</em>)', [
        '@menu' => $menu_label,
        '@link' => $menu_link_label,
      ]),
      '#default_value' => $mega_menu->getStatus(),
      '#description' => $this->t('<strong>Important:</strong> The mega menu only works for <em>top-level parent menu items</em>. If this item is not a top-level parent, enabling this option will have no effect — the mega menu functionality will be ignored even if configured. Please ensure this menu item is a top-level parent to use the mega menu.'),
    ];

    // Main wrapper for AJAX replacement.
    $form['rows_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'rows-wrapper'],
      '#states' => [
        'visible' => [
          ':input[name="enable_mega_menu"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Form-level help text for context.
    $form['rows_wrapper']['description'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<p>Use the sections below to build the layout and content of your mega menu. Each **Section (Row)** is a horizontal area, which can hold multiple **Columns** (vertical containers), and each **Column** can contain multiple **Blocks (Content)**.</p>'),
    ];

    // Load from Config if FormState is empty (First load).
    if (!$form_state->has('layout')) {
      $form_state->set('layout', $mega_menu->getRows() ?: []);
    }

    $layout = $form_state->get('layout');

    $form['rows_wrapper']['canvas'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'mega-menu-canvas'],
    ];

    $form['rows_wrapper']['canvas']['add_row'] = [
      '#type' => 'submit',
      '#value' => $this->t('＋ Add New Row'),
      '#submit' => ['::addRow'],
      '#ajax' => [
        'callback' => '::refreshAjax',
        'wrapper' => 'mega-menu-canvas',
        'progress' => [
    // This triggers the fullscreen overlay.
          'type' => 'fullscreen',
          'message' => $this->t('Creating Row...'),
        ],
      ],
      '#attributes' => ['class' => ['button', 'add-new-section-button']],
    ];

    foreach ($layout as $r_key => $row) {
      $row_num = $r_key + 1;

      $form['rows_wrapper']['canvas']['rows'][$r_key] = [
        '#type' => 'details',
        '#title' => $this->t('ROW #@num', ['@num' => $row_num]),
        '#open' => TRUE,
        '#attributes' => ['class' => ['mega-menu-row-item'], 'data-id' => 'row-' . $r_key],
      ];

      $current_mode = $row['column_layout_mode'] ?? 'auto';

      $form['rows_wrapper']['canvas']['rows'][$r_key]['column_layout_mode'] = [
        '#type' => 'select',
        '#title' => $this->t('Column Width Mode for Row #@r_key', ['@r_key' => $r_key + 1]),
          // Field Help Text.
        '#description' => $this->t('Choose how the columns in this section should be sized.'),
        '#options' => $layout_options + ['custom' => $this->t('Custom (Add individual CSS classes to row and columns for custom layout and styling)')],
        '#default_value' => $current_mode,
        '#prefix' => '<div class="row-layout-mode-container">',
        '#suffix' => '</div>',
        '#attributes' => [
          'id' => "edit-column-layout-mode-{$r_key}",
          'data-row-key' => $r_key,
        ],
      ];

      $form['rows_wrapper']['canvas']['rows'][$r_key]['single_row_class'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Row #@c_key class', ['@c_key' => $r_key + 1]),
        '#default_value' => $row['single_row_class'] ?? '',
        '#description' => $this->t('Enter a CSS class to style this row. You can add multiple classes, separated by spaces.'),
        '#states' => [
          'visible' => [
            ":input[id='edit-column-layout-mode-{$r_key}']" => ['value' => 'custom'],
          ],
        ],
      ];

      $form['rows_wrapper']['canvas']['rows'][$r_key]['actions'] = [
        '#type' => 'container',
      // Use a class instead of inline style.
        '#attributes' => ['class' => ['row-actions-wrapper']],
        'add_column' => [
          '#type' => 'submit',
          '#value' => $this->t('＋ Add Column'),
          '#name' => "add_col_$r_key",
          '#submit' => ['::addColumn'],
          '#ajax' => [
            'callback' => '::refreshAjax',
            'wrapper' => 'mega-menu-canvas',
            'progress' => [
      // This triggers the fullscreen overlay.
              'type' => 'fullscreen',
              'message' => $this->t('Creating Column...'),
            ],
          ],
          '#attributes' => ['class' => ['button', 'button--secondary', 'add-new-section-button']],
        ],
        'remove_row' => [
          '#type' => 'submit',
          '#value' => $this->t('⚠ Remove Row #@r_key', ['@r_key' => $r_key + 1]),
          '#name' => "rm_row_$r_key",
          '#submit' => ['::removeRow'],
          '#ajax' => [
            'callback' => '::refreshAjax',
            'wrapper' => 'mega-menu-canvas',
            'progress' => [
        // This triggers the fullscreen overlay.
              'type' => 'fullscreen',
              'message' => $this->t('Removing Row...'),
            ],
          ],
          '#attributes' => [
            'class' => ['button--danger', 'remove-section-button'],
          ],
        ],
      ];

      $form['rows_wrapper']['canvas']['rows'][$r_key]['columns'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['mega-flex-wrapper']],
      ];

      foreach ($row['columns'] ?? [] as $c_key => $column) {
        $col_num = $c_key + 1;

        $form['rows_wrapper']['canvas']['rows'][$r_key]['columns'][$c_key] = [
          '#type' => 'details',
          '#title' => $this->t('Column #@num(Row #@r_key)', ['@num' => $col_num, '@r_key' => $r_key + 1]),
          '#open' => $column['blocks'] ? FALSE : TRUE,
          '#attributes' => [
            'class' => ['mega-flex-column', 'sortable-blocks-list'],
            'data-col-index' => $c_key,
            'data-id' => 'row-' . $r_key . '-' . $c_key,
          ],
        ];

        $form['rows_wrapper']['canvas']['rows'][$r_key]['columns'][$c_key]['col_actions'] = [
          '#type' => 'container',
          'remove_column' => [
            '#type' => 'submit',
            '#value' => $this->t('✕'),
            '#name' => "rm_col_{$r_key}_{$c_key}",
            '#submit' => ['::removeColumn'],
            '#ajax' => [
              'callback' => '::refreshAjax',
              'wrapper' => 'mega-menu-canvas',
              'progress' => [
        // This triggers the fullscreen overlay.
                'type' => 'fullscreen',
                'message' => $this->t('Removing Column...'),
              ],
            ],
            '#attributes' => ['class' => ['remove-column-button'], 'title' => 'Remove this column'],
          ],
        ];

        $form['rows_wrapper']['canvas']['rows'][$r_key]['columns'][$c_key]['col_class'] = [
          '#type' => 'container',
          'column_class' => [
            '#type' => 'textfield',
            '#title' => $this->t('Column #@c_key class', ['@c_key' => $c_key + 1]),
            '#default_value' => $column['col_class'] ?? '',
            '#description' => $this->t('Enter a CSS class to style this column. You can add multiple classes, separated by spaces.'),
          ],
          '#states' => [
            'visible' => [
              ":input[id='edit-column-layout-mode-{$r_key}']" => ['value' => 'custom'],
            ],
          ],
        ];

        $form['rows_wrapper']['canvas']['rows'][$r_key]['columns'][$c_key]['blocks_container'] = [
          '#type' => 'container',
          '#attributes' => ['class' => ['blocks-drop-zone']],
        ];

        foreach ($column['blocks'] ?? [] as $b_key => $block) {
          $select_id = Html::getUniqueId("sel-$r_key-$c_key-$b_key");

          $form['rows_wrapper']['canvas']['rows'][$r_key]['columns'][$c_key]['blocks_container'][$b_key] = [
            '#type' => 'details',
            '#title' => $this->t('Block Section #@b_key', ['@b_key' => $b_key + 1]),
            '#attributes' => [
              'class' => [
                'draggable-block-item',
              ],
              'data-id' => 'row-' . $r_key . '-' . $c_key . '-' . $b_key,
            ],
            '#open' => TRUE,
          ];

          $b_ref = &$form['rows_wrapper']['canvas']['rows'][$r_key]['columns'][$c_key]['blocks_container'][$b_key];

          $b_ref['category'] = [
            '#type' => 'select',
            '#title' => $this->t('Block Category'),
            '#options' => [
              'content_id' => $this->t('Content Block'),
              'view_id'    => $this->t('View Block'),
              'theme_id'   => $this->t('Theme/System Block'),
            ],
            '#default_value' => $block['category'] ?? 'content_id',
            '#description' => $this->t('Select the source of the content to display.'),
            '#attributes' => ['id' => $select_id],
          ];

          // Source selections with saved default values.
          $b_ref['content_id'] = [
            '#type' => 'select',
            '#options' => $form_state->get('opt_content'),
            '#description' => $this->t('Select Content Block'),
            '#default_value' => $block['content_id'] ?? '',
            '#states' => ['visible' => [":input[id='$select_id']" => ['value' => 'content_id']]],
          ];

          $b_ref['view_id'] = [
            '#type' => 'select',
            '#options' => $form_state->get('opt_views'),
            '#description' => $this->t('Select View Block'),
            '#default_value' => $block['view_id'] ?? '',
            '#states' => ['visible' => [":input[id='$select_id']" => ['value' => 'view_id']]],
          ];

          $b_ref['theme_id'] = [
            '#type' => 'select',
            '#options' => $form_state->get('opt_theme'),
            '#description' => $this->t('Select Theme Block'),
            '#default_value' => $block['theme_id'] ?? '',
            '#states' => ['visible' => [":input[id='$select_id']" => ['value' => 'theme_id']]],
          ];

          $b_ref['show_label'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Show Label'),
            '#description' => $this->t('Display the block title above the content.'),
            '#default_value' => $block['show_label'] ?? FALSE,
          ];

          $b_ref['row_id'] = [
            '#type' => 'hidden',
            '#access' => FALSE,
            '#default_value' => $row_num,
            '#attributes' => [
              'class' => [
                'sync-row-id',
              ],
            ],
          ];
          $b_ref['column_id'] = [
            '#type' => 'hidden',
            '#access' => FALSE,
            '#default_value' => $col_num,
            '#attributes' => [
              'class' => [
                'sync-col-id',
              ],
            ],
          ];
          $b_ref['weight'] = [
            '#type' => 'hidden',
            '#access' => FALSE,
            '#default_value' => $b_key,
            '#attributes' => [
              'class' => [
                'sync-weight',
              ],
            ],
          ];

          $b_ref['remove'] = [
            '#type' => 'submit',
            '#value' => $this->t('⚠ Remove Block'),
            '#name' => "rm_blk_{$r_key}_{$c_key}_{$b_key}",
            '#submit' => ['::removeBlock'],
            '#ajax' => [
              'callback' => '::refreshAjax',
              'wrapper' => 'mega-menu-canvas',
              'progress' => [
          // This triggers the fullscreen overlay.
                'type' => 'fullscreen',
                'message' => $this->t('Removing Block...'),
              ],
            ],
            '#attributes' => ['class' => ['button--small', 'remove-section-button']],
          ];
        }

        $form['rows_wrapper']['canvas']['rows'][$r_key]['columns'][$c_key]['add_block'] = [
          '#type' => 'submit',
          '#value' => $this->t('Click here to add section'),
          '#name' => "add_blk_{$r_key}_{$c_key}",
          '#attributes' => [
            'class' => ['add-section-button'],
          ],
          '#submit' => ['::addBlock'],
          '#ajax' => [
            'callback' => '::refreshAjax',
            'wrapper' => 'mega-menu-canvas',
            'progress' => [
          // This triggers the fullscreen overlay.
              'type' => 'fullscreen',
              'message' => $this->t('Adding Block...'),
            ],
          ],
        ];
      }
    }

    // ==========================================================================
    // Menu Display Settings (Mega Menu Builder)
    // ==========================================================================
    $form['rows_wrapper']['menu_display_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Menu display settings'),
      '#description' => $this->t('Configure menu items, mega menu rows, icons, and accessibility settings.'),
      '#open' => TRUE,
      '#attributes' => ['data-id' => 'row-display-settings'],
      '#states' => [
        'visible' => [
          ':input[name="enable_mega_menu"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // ==========================================================================
    // Section: Menu content & layout
    // ==========================================================================
    $form['rows_wrapper']['menu_display_settings']['content_layout'] = [
      '#type' => 'details',
      '#title' => $this->t('Menu content & layout'),
      '#description' => $this->t('Configure what appears in the mega menu and how menu items are arranged.'),
      '#open' => TRUE,
    ];

    $form['rows_wrapper']['menu_display_settings']['content_layout']['menu_display_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Menu display mode'),
      '#description' => $this->t('Choose whether to display menu items, mega menu rows, or both.'),
      '#options' => [
        'menu_only' => $this->t('Menu items only'),
        'menu_and_mega' => $this->t('Menu items + mega menu rows'),
        'mega_only' => $this->t('Mega menu rows only'),
      ],
      '#default_value' => $display_settings['menu_display_mode'] ?? 'mega_only',
    ];

    $form['rows_wrapper']['menu_display_settings']['content_layout']['menu_items_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Menu items position'),
      '#description' => $this->t('Position menu items above or below mega menu rows when both are displayed.'),
      '#options' => [
        'top' => $this->t('Above mega menu rows'),
        'bottom' => $this->t('Below mega menu rows'),
      ],
      '#default_value' => $display_settings['menu_items_position'] ?? 'bottom',
      '#states' => [
        'visible' => [
          ':input[name="rows_wrapper[menu_display_settings][content_layout][menu_display_mode]"]' => [
        ['value' => 'menu_and_mega'],
          ],
        ],
      ],
    ];

    $form['rows_wrapper']['menu_display_settings']['content_layout']['menu_items_layout'] = [
      '#type' => 'select',
      '#title' => $this->t('Menu items layout'),
      '#description' => $this->t('Define the column width of each menu item. Smaller fractions create more columns.'),
      '#options' => $layout_options,
      '#default_value' => $display_settings['menu_items_layout'] ?? '1-4',
      '#states' => [
        'visible' => [
          ':input[name="rows_wrapper[menu_display_settings][content_layout][menu_display_mode]"]' => [
        ['value' => 'menu_only'],
        ['value' => 'menu_and_mega'],
          ],
        ],
      ],
    ];

    // ==========================================================================
    // Section: CSS classes
    // ==========================================================================
    $default_wrapper_class = advanced_mega_menu_sanitize_classes($display_settings['wrapper_class'] ?? '');
    $default_row_class = advanced_mega_menu_sanitize_classes($display_settings['row_class'] ?? '');
    $default_column_class = advanced_mega_menu_sanitize_classes($display_settings['column_class'] ?? '');

    $form['rows_wrapper']['menu_display_settings']['css_classes'] = [
      '#type' => 'details',
      '#title' => $this->t('Common CSS classes'),
      '#description' => $this->t('Add custom classes to target different parts of the mega menu.'),
      '#open' => FALSE,
    ];

    $form['rows_wrapper']['menu_display_settings']['css_classes']['wrapper_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Mega menu wrapper class'),
      '#description' => $this->t('Classes applied to the outer wrapper of the mega menu.'),
      '#default_value' => $default_wrapper_class,
    ];

    $form['rows_wrapper']['menu_display_settings']['css_classes']['row_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Menu row class'),
      '#description' => $this->t(
        'Classes applied to rows containing menu items or mega menu rows.<br/><em>These classes are used as a fallback when no custom class is defined for an individual row.</em>'
      ),
      '#default_value' => $default_row_class,
    ];

    $form['rows_wrapper']['menu_display_settings']['css_classes']['column_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Menu column class'),
      '#description' => $this->t(
        'Classes applied to individual menu columns inside the mega menu.<br/><em>These classes are used as a fallback when no custom class is defined for an individual column in row.</em>'
      ),
      '#default_value' => $default_column_class,
    ];

    $request = $this->requestStack->getCurrentRequest();
    $is_modal = $request->query->get('_wrapper_format');

    if ($is_modal) {
      $form['actions']['cancel'] = [
        '#type' => 'button',
        '#value' => $this->t('Cancel'),
        '#attributes' => ['class' => ['button-cancel-configuration', 'button']],
        '#weight' => 100,
      ];
    }

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save Configuration'),
      '#attributes' => ['class' => ['button-save-configuration', 'button']],
    ];

    // Return parent::buildForm($form, $form_state);.
    return $form;
  }

  /**
   * AJAX callback that refreshes the mega menu canvas and scrolls to new items.
   *
   * Replaces the mega menu canvas after an add operation (row, column, or
   * block) and determines the appropriate DOM selector to scroll into view
   * based on the triggering form element.
   *
   * @param array $form
   *   The complete form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   An AJAX response that updates the canvas and optionally scrolls to
   *   the newly added element.
   */
  public function refreshAjax($form, $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#mega-menu-canvas', $form['rows_wrapper']['canvas']));

    $trigger = $form_state->getTriggeringElement();
    $name = $trigger['#name'] ?? '';
    $selector = '';

    // 1. If adding a ROW: Target the last row in the canvas
    if (str_contains($name, 'op')) {
      $selector = '#mega-menu-canvas .mega-menu-row-item:last-child';
    }

    // 2. If adding a COLUMN: Target the last column INSIDE that specific row
    elseif (preg_match('/add_col_(\d+)/', $name, $matches)) {
      $row_index = $matches[1];
      // We target the row by its data-id attribute (which you set in buildForm)
      $selector = '[data-id="row-' . $row_index . '"] .mega-flex-column:last-child';
    }

    // 3. If adding a BLOCK: Target the last block INSIDE that specific column
    elseif (preg_match('/add_blk_(\d+)_(\d+)/', $name, $matches)) {
      $row_index = $matches[1];
      $col_index = $matches[2];
      $selector = '[data-id="row-' . $row_index . '-' . $col_index . '"] .draggable-block-item:last-child';
    }

    if ($selector) {
      $response->addCommand(new ScrollToCommand($selector));
    }

    return $response;
  }

  /**
   * Adds a new row to the mega menu layout.
   *
   * Updates the form state layout array and marks the form for rebuild.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function addRow($form, $form_state) {
    $layout = $form_state->get('layout');
    $layout[] = ['columns' => []];
    $form_state->set('layout', $layout);
    $form_state->setRebuild();
  }

  /**
   * Removes a row from the mega menu layout.
   *
   * Determines the row to remove from the triggering element's name,
   * updates the form state, and rebuilds the form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function removeRow($form, $form_state) {
    $r_key = str_replace('rm_row_', '', $form_state->getTriggeringElement()['#name']);
    $layout = $form_state->get('layout');
    unset($layout[$r_key]);
    $form_state->set('layout', array_values($layout));
    $form_state->setRebuild();
  }

  /**
   * Adds a new column to a specific row in the mega menu layout.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function addColumn($form, $form_state) {
    $r_key = str_replace('add_col_', '', $form_state->getTriggeringElement()['#name']);
    $layout = $form_state->get('layout');
    $layout[$r_key]['columns'][] = ['blocks' => []];
    $form_state->set('layout', $layout);
    $form_state->setRebuild();
  }

  /**
   * Removes a column from a specific row in the mega menu layout.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function removeColumn($form, $form_state) {
    preg_match('/rm_col_(\d+)_(\d+)/', $form_state->getTriggeringElement()['#name'], $m);
    $layout = $form_state->get('layout');
    unset($layout[$m[1]]['columns'][$m[2]]);
    $layout[$m[1]]['columns'] = array_values($layout[$m[1]]['columns']);
    $form_state->set('layout', $layout);
    $form_state->setRebuild();
  }

  /**
   * Adds a new block to a specific column in the mega menu layout.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function addBlock($form, $form_state) {
    preg_match('/add_blk_(\d+)_(\d+)/', $form_state->getTriggeringElement()['#name'], $m);
    $layout = $form_state->get('layout');
    $layout[$m[1]]['columns'][$m[2]]['blocks'][] = ['text' => '', 'category' => 'content'];
    $form_state->set('layout', $layout);
    $form_state->setRebuild();
  }

  /**
   * Removes a block from a specific column in the mega menu layout.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function removeBlock($form, $form_state) {
    preg_match('/rm_blk_(\d+)_(\d+)_(\d+)/', $form_state->getTriggeringElement()['#name'], $m);
    $layout = $form_state->get('layout');
    unset($layout[$m[1]]['columns'][$m[2]]['blocks'][$m[3]]);
    $layout[$m[1]]['columns'][$m[2]]['blocks'] = array_values($layout[$m[1]]['columns'][$m[2]]['blocks']);
    $form_state->set('layout', $layout);
    $form_state->setRebuild();
  }

  /**
   * Prepares available content, views, and theme block options for selection.
   *
   * Loads block content, views blocks, and theme blocks and stores them
   * in the form state for later use in select elements.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  protected function prepareSourceOptions($form_state) {
    if ($form_state->get('opt_content')) {
      return;
    }

    // Wrap block content loading in try/catch.
    try {
      // Load all block content entities.
      $blocks = $this->entityTypeManager->getStorage('block_content')->loadMultiple();
      $block_types = $this->entityTypeManager->getStorage('block_content_type')->loadMultiple();

      // Prepare grouped options array.
      $opt_content = ['' => '- Select -'];

      foreach ($blocks as $block) {
        /** @var \Drupal\block_content\Entity\BlockContent $block */
        $bundle = $block->bundle();
        $bundle_label = $block_types[$bundle]->label() ?? $bundle;

        // Initialize group if needed.
        if (!isset($opt_content[$bundle_label])) {
          $opt_content[$bundle_label] = [];
        }

        $opt_content[$bundle_label][$block->id()] = $block->label();
      }

      // Optional: sort groups and items alphabetically.
      ksort($opt_content);
      foreach ($opt_content as &$group) {
        if (is_array($group)) {
          asort($group);
        }
      }
      $form_state->set('opt_content', $opt_content);

      // Views Blocks.
      $opt_views = ['' => '- Select -'];
      foreach (Views::getAllViews() as $id => $view) {
        foreach ($view->get('display') as $did => $d) {
          if ($d['display_plugin'] == 'block') {
            $opt_views[$view->label()]["$id:$did"] = $d['display_title'];
          }
        }
      }
      $form_state->set('opt_views', $opt_views);

      // Theme Blocks.
      $theme = $this->configFactory->get('system.theme')->get('default');
      $opt_theme = ['' => '- Select -'];
      foreach ($this->entityTypeManager->getStorage('block')->loadByProperties(['theme' => $theme]) as $tb) {
        $opt_theme[$tb->getRegion()][$tb->id()] = $tb->label();
      }
      $form_state->set('opt_theme', $opt_theme);
    }
    catch (\Throwable $e) {
      $this->logger->error('Failed to load block content options: @message',
        [
          '@message' => $e->getMessage(),
        ]
      );
      $form_state->set('opt_content', ['' => '- Select -']);
    }
  }

  /**
   * Handles submission of the Mega Menu configuration form.
   *
   * Processes form values for rows, columns, and blocks layout, as well as
   * display settings and status, then saves the corresponding Mega Menu
   * entity. Provides success or error messages to the user.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // 1. Retrieve the entity from Form State
    /** @var \Drupal\advanced_mega_menu\Entity\MegaMenuContent $mega_menu */
    $mega_menu = $form_state->get('mega_menu_entity');

    if (!$mega_menu) {
      $this->messenger()->addError($this->t('Entity lost during form submission. Please try reloading the page.'));
      return;
    }

    // 2. Extract Values using the exact Form API Tree paths
    $values = $form_state->getValues();

    // Enable/Disable.
    $enable_mega_menu = $values['enable_mega_menu'] ?? 0;

    // Layout Rows (Rows -> Columns -> Blocks)
    $rows_input = $values['rows_wrapper']['canvas']['rows'] ?? [];
    $clean_layout = [];

    $row_count = 0;

    foreach ($rows_input as $row_data) {
      if ($row_count >= self::MAX_ROWS) {
        continue;
      }

      $row_item = [
        'column_layout_mode' => $row_data['column_layout_mode'] ?? 'auto',
        'single_row_class' => $row_data['single_row_class'] ?? '',
        'columns' => [],
      ];

      if (!empty($row_data['columns'])) {
        $column_count = 0;

        foreach ($row_data['columns'] as $col_data) {
          if ($column_count >= self::MAX_COLUMNS_PER_ROW) {
            continue;
          }

          $col_item = [
          // Note the extra nesting here.
            'col_class' => $col_data['col_class']['column_class'] ?? '',
            'blocks' => [],
          ];

          if (!empty($col_data['blocks_container'])) {
            $blocks_count = 0;

            $blocks = $col_data['blocks_container'];
            // Sort by weight.
            uasort($blocks, function ($a, $b) {
                return (int) ($a['weight'] ?? 0) <=> (int) ($b['weight'] ?? 0);
            });

            foreach ($blocks as $b_vals) {
              if ($blocks_count >= self::MAX_BLOCKS_PER_COLUMN) {
                continue;
              }
              // Clean up the block data before saving.
              unset($b_vals['remove']);
              $col_item['blocks'][] = $b_vals;
              $blocks_count++;
            }
          }
          $row_item['columns'][] = $col_item;
          $column_count++;
        }
      }
      $clean_layout[] = $row_item;
      $row_count++;
    }

    // 3. Extract Display Settings (Correcting the deep nesting)
    $mds = $values['rows_wrapper']['menu_display_settings'] ?? [];

    $display_settings = [
      'menu_display_mode'   => $mds['content_layout']['menu_display_mode'] ?? 'mega_only',
      'menu_items_position' => $mds['content_layout']['menu_items_position'] ?? 'bottom',
      'menu_items_layout'   => $mds['content_layout']['menu_items_layout'] ?? '1-4',
      'wrapper_class'       => $mds['css_classes']['wrapper_class'] ?? '',
      'row_class'           => $mds['css_classes']['row_class'] ?? '',
      'column_class'        => $mds['css_classes']['column_class'] ?? '',
    ];

    // 4. Final Save
    try {
      $mega_menu->setRows($clean_layout);
      $mega_menu->setStatus((bool) $enable_mega_menu);
      // Important: Use the key 'menu_display_settings' to match the
      // buildForm's $mega_menu->getDisplaySettings call.
      $mega_menu->setDisplaySettings($display_settings);
      $mega_menu->save();

      $this->messenger()->addStatus($this->t('Mega Menu configuration saved successfully.'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to save: @message', ['@message' => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // 1. Validate global display setting classes.
    // We use strings as keys here to avoid "Illegal Offset" errors.
    $class_configs = [
      'wrapper_class' => 'rows_wrapper][menu_display_settings][css_classes][wrapper_class',
      'row_class'     => 'rows_wrapper][menu_display_settings][css_classes][row_class',
      'column_class'  => 'rows_wrapper][menu_display_settings][css_classes][column_class',
    ];

    foreach ($class_configs as $field_key => $element_path) {
      // Create the path array dynamically for getValue().
      $value_path = ['rows_wrapper', 'menu_display_settings', 'css_classes', $field_key];

      $this->validateClassField(
        $form_state,
        $value_path,
        $element_path,
        $this->t('The @field contains invalid characters or starts with a number.', [
          '@field' => str_replace('_', ' ', $field_key),
        ])
      );
    }

    // 2. Validate Per-Column classes inside the Canvas.
    $rows = $form_state->getValue(['rows_wrapper', 'canvas', 'rows']) ?? [];

    // Rows limit.
    if (count($rows) > self::MAX_ROWS) {
      $form_state->setErrorByName(
        'rows_wrapper][canvas][rows',
        $this->t(
          'The mega menu supports a maximum of @max rows. You have added @count.',
          [
            '@max' => self::MAX_ROWS,
            '@count' => count($rows),
          ]
        )
      );
    }

    foreach ($rows as $r_key => $row) {
      $single_row_class = $row['single_row_class'] ?? '';
      if ($single_row_class && !advanced_mega_menu_are_valid_classes($single_row_class)) {
        $form_state->setErrorByName(
          "rows_wrapper][canvas][rows][$r_key][single_row_class",
          $this->t(
            'Row @row contains invalid CSS classes.',
            [
              '@row' => $r_key + 1,
            ]
          )
        );
      }

      // Columns limit per row.
      $columns = $row['columns'] ?? [];
      if (count($columns) > self::MAX_COLUMNS_PER_ROW) {
        $form_state->setErrorByName(
          "rows_wrapper][canvas][rows][$r_key][columns",
          $this->t(
            'Row @row supports a maximum of @max columns. You have added @count.',
            [
              '@row' => $r_key + 1,
              '@max' => self::MAX_COLUMNS_PER_ROW,
              '@count' => count($columns),
            ]
          )
        );
      }

      foreach ($columns as $c_key => $column) {
        // According to your buildForm, the structure is:
        // [col_class][column_class].
        $column_class = $column['col_class']['column_class'] ?? '';

        if ($column_class && !advanced_mega_menu_are_valid_classes($column_class)) {
          $form_state->setErrorByName(
            "rows_wrapper][canvas][rows][$r_key][columns][$c_key][col_class][column_class",
            $this->t(
              'Column @column in row @row contains invalid CSS classes.',
              [
                '@row' => $r_key + 1,
                '@column' => $c_key + 1,
              ]
            )
          );
        }

        // Blocks limit per column.
        $blocks = $column['blocks_container'] ?? [];
        if (count($blocks) > self::MAX_BLOCKS_PER_COLUMN) {
          $form_state->setErrorByName(
            "rows_wrapper][canvas][rows][$r_key][columns][$c_key][blocks_container",
            $this->t(
              'Column @column in row @row supports a maximum of @max blocks. You have added @count.',
              [
                '@row' => $r_key + 1,
                '@column' => $c_key + 1,
                '@max' => self::MAX_BLOCKS_PER_COLUMN,
                '@count' => count($blocks),
              ]
            )
          );
        }
      }
    }
  }

  /**
   * Validates a CSS class field.
   */
  private function validateClassField(
    FormStateInterface $form_state,
    array $value_path,
    string $error_name,
    string $message,
  ): void {
    $value = $form_state->getValue($value_path);

    if ($value && !advanced_mega_menu_are_valid_classes($value)) {
      $form_state->setErrorByName($error_name, $message);
    }
  }

}
