<?php

namespace Drupal\advanced_mega_menu;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Render\Markup;

/**
 * Provides a list builder for MegaMenuContent entities.
 */
class MegaMenuContentListBuilder extends ConfigEntityListBuilder {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a MegaMenuContentListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct($entity_type, $storage, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($entity_type, $storage);
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('config.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['label'] = $this->t('Mega Menu Content');
    $header['menu_id'] = $this->t('Menu');
    $header['status'] = $this->t('Status');
    $header['operations'] = $this->t('Operations');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow($entity): array {
    $row['label'] = $entity->label();

    // Load enabled menus from config.
    $config = $this->configFactory->get('advanced_mega_menu.settings');
    $enabled_menus = $config->get('enabled_menus') ?: [];

    // Convert menu_id into a clickable link if the menu exists.
    $menu_id = $entity->getMenuId();

    if (in_array($menu_id, $enabled_menus)) {
      $enabled = $this->t('Mega Menu Enabled');
    }
    else {
      $enabled = $this->t('Mega Menu Disabled');
    }

    $menu = $this->entityTypeManager->getStorage('menu')->load($menu_id);
    if ($menu) {
      $url = Url::fromRoute('entity.menu.edit_form', ['menu' => $menu_id]);
      // Create the link string.
      $link = Link::fromTextAndUrl($menu->label(), $url)->toString();

      // Append the $enabled message.
      $row['menu_id'] = Markup::create($link . ' (<small><em>' . $enabled . '</em></small>)');
    }
    else {
      // Menu doesn't exist, show plain text.
      $row['menu_id'] = $menu_id;
    }

    $row['status'] = $entity->getStatus() ? $this->t('Enabled') : $this->t('Disabled');

    // Operations: edit via your custom route and delete via entity delete form.
    $row['operations']['data'] = [
      '#type' => 'operations',
      '#links' => [
        'edit' => [
          'title' => $this->t('Edit'),
          'url' => Url::fromRoute('advanced_mega_menu.action', [
            'menu_id' => $menu_id,
            'menu_link_content' => $entity->getFormId(),
          ]),
        ],
        'delete' => [
          'title' => $this->t('Delete'),
          'url' => Url::fromRoute('entity.megamenu_content.delete_form', [
            'megamenu_content' => $entity->id(),
          ]),
        ],
      ],
    ];

    return $row + parent::buildRow($entity);
  }

}
