<?php

namespace Drupal\advanced_mega_menu\Menu;

use Drupal\Core\Menu\MenuLinkTreeInterface;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Helper service for menu link operations.
 */
class MenuLinkHelper {

  /**
   * Menu link tree service.
   *
   * @var \Drupal\Core\Menu\MenuLinkTreeInterface
   */
  protected MenuLinkTreeInterface $menuLinkTree;

  /**
   * Constructs the helper.
   */
  public function __construct(MenuLinkTreeInterface $menu_link_tree) {
    $this->menuLinkTree = $menu_link_tree;
  }

  /**
   * Get all menu link keys with their titles.
   *
   * @param string $menu_name
   *   Menu machine name.
   * @param int|null $max_depth
   *   Maximum depth, NULL for unlimited.
   *
   * @return array
   *   Flat array: ['key' => string, 'title' => string].
   */
  public function getMenuLinksWithTitles(string $menu_name, ?int $max_depth = NULL): array {
    $parameters = (new MenuTreeParameters())
      ->setMaxDepth($max_depth);

    $tree = $this->menuLinkTree->load($menu_name, $parameters);

    $tree = $this->menuLinkTree->transform($tree, [
      ['callable' => 'menu.default_tree_manipulators:checkAccess'],
      ['callable' => 'menu.default_tree_manipulators:generateIndexAndSort'],
    ]);

    $links = [];

    $flattenTree = function (array $tree) use (&$flattenTree, &$links) {
      foreach ($tree as $element) {
        $plugin_id = $element->link->getPluginId();
        $definition = $element->link->getPluginDefinition();

        $title = $definition['title'] ?? $element->link->getTitle();

        if (str_starts_with($plugin_id, 'menu_link_content:')) {
          $plugin_id = substr($plugin_id, strlen('menu_link_content:'));
        }

        $links[] = [
          'key' => $plugin_id,
          'title' => $title instanceof TranslatableMarkup ? $title->render() : (string) $title,
        ];

        if (!empty($element->subtree)) {
          $flattenTree($element->subtree);
        }
      }
    };

    $flattenTree($tree);

    return $links;
  }

}
