<?php

namespace Drupal\advanced_mega_menu\Plugin\rest\resource;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Psr\Log\LoggerInterface;
use Drupal\advanced_mega_menu\Service\MegaMenuRowBuilder;

/**
 * Provides a REST resource for retrieving Advanced Mega Menu items.
 *
 * This REST resource allows external clients (or frontend applications)
 * to retrieve mega menu data for a specific menu and menu link plugin ID.
 *
 * Example request:
 *   GET /api/advanced-mega-menu/main/123
 *
 * Example response:
 * {
 *   "#theme": "advanced_mega_menu",
 *   "#mega_menu_contents": [...],
 *   "#menu_link_contents": [...],
 *   "#display_settings": {...},
 *   "#has_children": true
 * }
 *
 * @RestResource(
 *   id = "advanced_mega_menu_resource",
 *   label = @Translation("Advanced Mega Menu REST"),
 *   uri_paths = {
 *     "canonical" = "/api/advanced-mega-menu/{menu_id}/{plugin_id}"
 *   }
 * )
 */
class MegaMenuResource extends ResourceBase implements ContainerFactoryPluginInterface {

  /**
   * The MegaMenuRowBuilder service.
   *
   * Builds the mega menu render array for a given menu link.
   *
   * @var \Drupal\advanced_mega_menu\Service\MegaMenuRowBuilder
   */
  protected MegaMenuRowBuilder $rowBuilder;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * Constructs a MegaMenuResource object.
   *
   * @param array $configuration
   *   Plugin configuration array.
   * @param string $plugin_id
   *   The plugin ID for this REST resource instance.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param array $serializer_formats
   *   Supported serialization formats (e.g. json, xml).
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger service for this resource.
   * @param \Drupal\advanced_mega_menu\Service\MegaMenuRowBuilder $row_builder
   *   Service that builds mega menu rows.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service used to render render arrays to HTML markup.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    MegaMenuRowBuilder $row_builder,
    RendererInterface $renderer,
  ) {
    parent::__construct(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $serializer_formats,
      $logger
    );

    $this->rowBuilder = $row_builder;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('advanced_mega_menu'),
      $container->get('advanced_mega_menu.mega_menu_row_builder'),
      $container->get('renderer')
    );
  }

  /**
   * Responds to GET requests for a mega menu item.
   *
   * @param string $menu_id
   *   The menu machine name (e.g. "main", "footer").
   * @param string $plugin_id
   *   The menu link plugin ID (e.g. "menu_link_content:123").
   *
   * @return \Drupal\rest\ResourceResponse
   *   The REST response containing the mega menu render array.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   Thrown when no mega menu exists or it is disabled.
   */
  public function get(string $menu_id, string $plugin_id): ResourceResponse {

    // Build mega menu rows using the dedicated service.
    $rows = $this->rowBuilder->buildMenuItemMegaRows($menu_id, $plugin_id);

    if (empty($rows)) {
      throw new NotFoundHttpException('Mega menu not found or disabled.');
    }

    /** @var \Drupal\Core\Render\RendererInterface $renderer */
    $renderer = $this->renderer;

    // Render the render array to HTML (Markup object / string)
    $megamenu_html = $renderer->renderRoot($rows);

    // Wrap HTML in REST response (as a string)
    $response = new ResourceResponse([
      'megamenu' => $megamenu_html,
    ]);

    // Attach cache metadata so REST responses invalidate correctly.
    $cache = new CacheableMetadata();
    $cache->addCacheContexts([
      'url.path',
    ]);
    $cache->addCacheTags([
      // Invalidate when the mega menu content entity changes.
      'mega_menu_content:' . str_replace('menu_link_content:', '', $plugin_id),
      // Invalidate when the menu configuration changes.
      'config:system.menu.' . $menu_id,
    ]);

    $response->addCacheableDependency($cache);

    return $response;
  }

}
