<?php

namespace Drupal\advanced_mega_menu\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\views\Views;

/**
 * Service class for rendering mega menu blocks.
 *
 * This service provides helper methods to render different types of blocks
 * used in the Advanced Mega Menu module, including:
 * - Theme blocks
 * - Views displays
 * - Custom content blocks (BlockContent entities)
 *
 * It handles:
 * - Validating input configuration
 * - Loading entities via dependency injection
 * - Preparing render arrays
 * - Optional display of block or view labels
 * - Error handling and logging via the dedicated logger channel
 *
 * Services injected:
 * - \Drupal\Core\Entity\EntityTypeManagerInterface: For
 * loading and rendering entities
 * - \Drupal\Core\Logger\LoggerChannelFactoryInterface: For
 * logging errors
 *
 * Example usage:
 * @code
 *   $renderer = \Drupal::service('advanced_mega_menu.block_renderer');
 *   $render_array = $renderer->renderThemeBlock($decoded_config, TRUE);
 * @endcode
 */
class MegaMenuBlockRenderer {

  use StringTranslationTrait;

  /**
   * The entity type manager service.
   *
   * Used to load and render entities such as blocks, block content, and views.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger channel for the Advanced Mega Menu module.
   *
   * Used to log errors or exceptions during block rendering operations.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new MegaMenuBlockRenderer service.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service, used for loading and rendering entities.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service, used to create a module-specific logger
   *   channel.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger_factory->get('advanced_mega_menu');
  }

  /**
   * Render a theme block.
   */
  public function renderThemeBlock($block_id, bool $show_label): array {
    try {

      if (empty($block_id)) {
        throw new \InvalidArgumentException('Theme block ID is missing.');
      }

      $block = $this->entityTypeManager
        ->getStorage('block')
        ->load($block_id);

      if (!$block) {
        throw new \RuntimeException('Theme block not found.');
      }

      $block_render = $this->entityTypeManager
        ->getViewBuilder('block')
        ->view($block);

      $build = [
        '#type' => 'container',
        '#attributes' => ['class' => ['custom-block-wrapper']],
        'block' => $block_render,
      ];

      if ($show_label) {
        $block_label = $block->label();
        $build['label'] = [
          '#markup' => '<div class="block-label">' . $block_label . '</div>',
          '#weight' => -10,
        ];
      }

      return $build;

    }
    catch (\Throwable $e) {
      $this->logger->error('Theme block render failed: @message', [
        '@message' => $e->getMessage(),
        // 'config' => $decoded,
      ]);

      return [
        '#markup' => $this->t('Unable to render this block.'),
      ];
    }
  }

  /**
   * Render a Views display.
   */
  public function renderViewBlock(array $decoded, bool $show_label): array {
    try {
      if (empty($decoded['view_id']) || empty($decoded['display_id'])) {
        throw new \InvalidArgumentException('View ID or display ID missing.');
      }

      $view = Views::getView($decoded['view_id']);

      if (!$view) {
        throw new \RuntimeException('View not found.');
      }

      if (!$view->storage->status()) {
        throw new \RuntimeException('View is disabled.');
      }

      if (!$view->setDisplay($decoded['display_id'])) {
        throw new \RuntimeException('View display does not exist.');
      }

      // Prepare the render array for the view.
      $rendered_view = $view->render();

      if ($show_label) {
        // Get the view label (title)
        $view_label = $view->storage->label();

        // Add the view label to the render array if needed.
        $rendered_view['#prefix'] = '<div class="view-label">' . $view_label . '</div>';
      }

      return $rendered_view;
    }
    catch (\Throwable $e) {
      $this->logger->error('View block render failed: @message', [
        '@message' => $e->getMessage(),
        // 'config' => $decoded,
      ]);

      return [
        '#markup' => $this->t('Unable to render this view.'),
      ];
    }
  }

  /**
   * Render a custom block content entity.
   */
  public function renderContentBlock($block_id, bool $show_label): array {

    try {

      if (!$block_id) {
        throw new \InvalidArgumentException('Content block ID is missing.');
      }

      $block_content = $this->entityTypeManager->getStorage('block_content')->load($block_id);

      if (!$block_content) {
        throw new \RuntimeException('Content block not found.');
      }

      // Get the view of the block content.
      $rendered_block = $this->entityTypeManager
        ->getViewBuilder('block_content')
        ->view($block_content);

      if ($show_label) {
        // Get the block's label (title)
        $block_label = $block_content->label();

        // Add the block label to the render array.
        $rendered_block['#prefix'] = '<div class="block-label">' . $block_label . '</div>';
      }

      return $rendered_block;

    }
    catch (\Throwable $e) {
      $this->logger->error('Content block render failed: @message', [
        '@message' => $e->getMessage(),
        // 'config' => $decoded,
      ]);

      return [
        '#markup' => $this->t('Unable to render this content block.'),
      ];
    }
  }

}
