<?php

namespace Drupal\advanced_mega_menu\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Builds mega menu rows for a given menu and menu link plugin ID.
 *
 * This service is responsible for loading the MegaMenuContent entity
 * and converting it into a render array consumable by REST, blocks,
 * or controllers.
 */
class MegaMenuRowBuilder {

  /**
   * Entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a MegaMenuRowBuilder service.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Builds the mega menu render array for a single menu item.
   *
   * This method loads the MegaMenuContent entity associated with a menu link,
   * evaluates its display configuration, and prepares the render array used
   * to output the mega menu.
   *
   * If the menu display mode allows normal menu children, they are extracted
   * from the provided menu item and converted into renderable menu link
   * contents. When this happens, the original `below` element is cleared
   * to prevent duplicate rendering.
   *
   * The `$item` argument is optional. If it is not provided, the method will
   * still build mega menu rows, but normal menu children will not be processed
   * or altered.
   *
   * @param string $menu_name
   *   The machine name of the menu the link belongs to (e.g. 'main').
   * @param string $plugin_id
   *   The menu link plugin ID (e.g. "menu_link_content:123").
   * @param array|null $item
   *   (optional) The menu item render array.
   *   Passed by reference so the method may modify `$item['below']`
   *   when menu children are consumed by the mega menu.
   *
   * @return array|null
   *   A render array for the mega menu rows, or NULL if:
   *   - No MegaMenuContent entity exists for the menu item
   *   - The mega menu row is disabled
   */
  public function buildMenuItemMegaRows(string $menu_name, string $plugin_id, ?array &$item = NULL): ?array {
    // Normalize plugin ID.
    $plugin_id = str_replace('menu_link_content:', '', $plugin_id);

    $storage = $this->entityTypeManager->getStorage('megamenu_content');

    $entities = $storage->loadByProperties([
      'form_id' => $plugin_id,
      'menu_id' => $menu_name,
    ]);

    $mega_menu_row = reset($entities) ?: NULL;

    if (!$mega_menu_row || !$mega_menu_row->getStatus()) {
      return NULL;
    }

    $display_settings = $mega_menu_row->getDisplaySettings();
    $menu_display_mode = $display_settings['menu_display_mode'] ?? 'mega_only';

    $menu_link_contents = [];
    $mega_menu_contents = [];

    // Load mega menu rows unless restricted to menu-only mode.
    if ($menu_display_mode !== 'menu_only') {
      $mega_menu_contents = $mega_menu_row->getRowsData();
    }

    // Load normal menu children when allowed and available.
    if (
      $item !== NULL
      && ($menu_display_mode === 'menu_only' || $menu_display_mode === 'menu_and_mega')
      && !empty($item['below'])
    ) {
      $menu_tree = $item['below'];
      $menu_link_contents = advanced_mega_menu_get_child_items($menu_tree);

      // Prevent duplicate rendering of child menu items.
      $item['below'] = [];
    }

    $has_children = !empty($mega_menu_contents)
      || !empty($menu_link_contents)
      || ($item !== NULL && !empty($item['below']));

    // Sanitize CSS class settings.
    foreach (['row_class', 'column_class'] as $key) {
      if (!empty($display_settings[$key])) {
        $display_settings[$key] = advanced_mega_menu_sanitize_classes($display_settings[$key]);
      }
    }

    return [
      '#theme' => 'advanced_mega_menu',
      '#mega_menu_contents' => $mega_menu_contents,
      '#menu_link_contents' => $menu_link_contents,
      '#display_settings' => $display_settings,
      '#has_children' => $has_children,
    ];
  }

}
