<?php

namespace Drupal\Tests\advanced_mega_menu\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Tests the frontend menu interactions (Mobile toggle and Collision).
 *
 * @group advanced_mega_menu
 */
class AdvancedMegaMenuFrontendTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['system', 'advanced_mega_menu'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests the mobile toggle behavior (<= 992px).
   *
   * Why: Ensures that on small screens, clicking the icon toggles the 'active'
   * class instead of following a link or doing nothing.
   */
  public function testMobileToggle() {
    // 1. Set viewport to mobile size.
    $this->getSession()->resizeWindow(400, 800);

    // 2. Build a mock menu rendering or navigate to a page with the menu.
    $this->drupalGet('<front>');

    $page = $this->getSession()->getPage();
    $icon = $page->find('css', '.submenu-icon');
    $parent_item = $page->find('css', '.advanced-mega-menu-item');

    if ($icon) {
      // 3. Click the icon.
      $icon->click();

      // 4. Assert: The parent item should now have the 'active' class.
      $this->assertTrue($parent_item->hasClass('active'), 'Parent item is active on mobile click.');

      // 5. Click again to toggle off.
      $icon->click();
      $this->assertFalse($parent_item->hasClass('active'), 'Parent item is inactive after second click.');
    }
  }

  /**
   * Tests desktop collision detection (> 992px).
   *
   * Why: If a submenu is too far to the right, it should get the 'open-left'
   * class to prevent it from flowing off-screen.
   */
  public function testSubmenuCollision() {
    // 1. Set viewport to desktop size.
    $this->getSession()->resizeWindow(1200, 800);
    $this->drupalGet('<front>');

    $page = $this->getSession()->getPage();
    $item = $page->find('css', '.advanced-mega-menu-item.has-submenu');

    if ($item) {
      // 2. Mock a scenario where the item is at the far right of the screen.
      // We do this via JS because placing blocks precisely in tests is hard.
      $this->getSession()->executeScript("
        const el = document.querySelector('.advanced-mega-menu-item.has-submenu');
        el.style.position = 'absolute';
        el.style.right = '0px';
      ");

      // 3. Trigger 'mouseenter' (Desktop behavior).
      $item->mouseOver();

      // 4. Assert: 'open-left' class should be applied.
      $this->assertTrue($item->hasClass('open-left'), 'Collision detection applied open-left class.');
    }
  }

  /**
   * Tests Mega Menu Panel centering and shifting.
   *
   * How: Verifies that the panel's transform style is updated dynamically
   * when the panel would otherwise hit the viewport edge.
   */
  public function testMegaMenuPanelCentering() {
    $this->getSession()->resizeWindow(1200, 800);
    $this->drupalGet('<front>');

    $page = $this->getSession()->getPage();
    $mega_item = $page->find('css', '.advanced-mega-menu-item.mega');

    if ($mega_item) {
      // Trigger mouseEnter.
      $mega_item->mouseOver();

      // Check if the panel has the transform style applied.
      $panel = $page->find('css', '.advanced-mega-menu-panel');
      $style = $panel->getAttribute('style');

      $this->assertStringContainsString('translateX', $style, 'Mega panel transform style was applied.');
      $this->assertStringContainsString('display: flex', $style, 'Mega panel is displayed on hover.');

      // Trigger mouseLeave.
      // Move mouse away to parent.
      $mega_item->getParent()->mouseOver();
      $style_after = $panel->getAttribute('style');
      $this->assertStringNotContainsString('display: flex', $style_after, 'Mega panel hidden on mouse leave.');
    }
  }

}
