<?php

namespace Drupal\Tests\advanced_mega_menu\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\system\Entity\Menu;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\advanced_mega_menu\Controller\AdvancedMegaMenuController;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Kernel tests for the AdvancedMegaMenuController.
 *
 * This test class focuses on **Kernel-level testing** of the controller.
 * Kernel tests allow testing the controller's logic and integration with
 * Drupal's services, entities, and configuration **without a full web
 * browser**.
 *
 * What this test covers:
 * 1. **Menu existence validation**: Ensures the controller throws a 404
 *    when an invalid menu ID is passed.
 * 2. **Menu link validation**: Ensures the controller throws a 404
 *    when an invalid menu link ID is passed.
 * 3. **Mega menu toggle enable**: Ensures enabling a mega menu updates
 *    the `advanced_mega_menu.settings` configuration correctly.
 * 4. **Mega menu toggle disable**: Ensures disabling a mega menu updates
 *    the configuration correctly.
 *
 * Notes:
 * - This test uses **KernelTestBase** because we need entity schemas
 *   (`menu`, `menu_link_content`, `megamenu_content`) and config.
 * - Kernel tests are faster than functional tests because they do not
 *   require a web server or JavaScript.
 * - Routes like `entity.menu.collection` need modules like `menu_ui` enabled
 *   to exist in Kernel test context.
 *
 * @group advanced_mega_menu
 */
class AdvancedMegaMenuControllerTest extends KernelTestBase {

  /**
   * Modules to enable for this test.
   *
   * Required for entities, fields, routes, and controller services:
   * - system: provides menus and basic entities
   * - user: required for entity schema
   * - link: provides the "link" field type for menu links
   * - menu_ui: provides the "entity.menu.collection" route
   * - menu_link_content: provides the menu link content entity
   * - advanced_mega_menu: the module under test.
   *
   * @var array
   */
  protected static $modules = [
    'system',
    'user',
    'link',
    'menu_ui',
    'menu_link_content',
    'advanced_mega_menu',
  ];

  /**
   * The controller being tested.
   *
   * @var \Drupal\advanced_mega_menu\Controller\AdvancedMegaMenuController
   */
  protected AdvancedMegaMenuController $controller;

  /**
   * {@inheritdoc}
   *
   * Sets up entity schemas, configuration, and test data.
   */
  protected function setUp(): void {
    parent::setUp();

    // Install necessary entity schemas for testing.
    $this->installEntitySchema('menu');
    $this->installEntitySchema('menu_link_content');
    $this->installEntitySchema('megamenu_content');

    // Install configuration for system and advanced_mega_menu.
    $this->installConfig([
      'system',
      'advanced_mega_menu',
    ]);

    // Create a test menu.
    Menu::create([
      'id' => 'test_menu',
      'label' => 'Test Menu',
    ])->save();

    // Create a test menu link in the menu.
    MenuLinkContent::create([
      'title' => 'Test Link',
      'menu_name' => 'test_menu',
      'link' => ['uri' => 'internal:/'],
    ])->save();

    // Instantiate the controller from Drupal's service container.
    // Using the class resolver ensures all dependencies are injected.
    $this->controller = $this->container
      ->get('class_resolver')
      ->getInstanceFromDefinition(AdvancedMegaMenuController::class);
  }

  /**
   * Test that action() throws 404 when the menu does not exist.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   */
  public function testActionMenuNotFound(): void {
    // Expect a 404 NotFound exception.
    $this->expectException(NotFoundHttpException::class);

    // Call the controller action with an invalid menu ID.
    $this->controller->action('invalid_menu', 'invalid');
  }

  /**
   * Test that action() throws 404 when the menu link does not exist.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   */
  public function testActionInvalidMenuLink(): void {
    // Expect a 404 NotFound exception.
    $this->expectException(NotFoundHttpException::class);

    // Call the controller action with a valid menu but invalid link ID.
    $this->controller->action('test_menu', 'invalid');
  }

  /**
   * Test that toggle() correctly enables a menu in configuration.
   *
   * After calling toggle() with 'enable', the menu ID should appear
   * in the 'enabled_menus' configuration.
   */
  public function testToggleEnable(): void {
    // Enable the test menu.
    $this->controller->toggle('test_menu', 'enable');

    // Fetch the updated configuration.
    $enabled = $this->config('advanced_mega_menu.settings')
      ->get('enabled_menus');

    // Assert that the menu is now enabled.
    $this->assertContains('test_menu', $enabled);
  }

  /**
   * Test that toggle() correctly disables a menu in configuration.
   *
   * First, set the menu as enabled, then disable it.
   * After calling toggle() with 'disable', the menu ID should be removed
   * from the 'enabled_menus' configuration.
   */
  public function testToggleDisable(): void {
    // Pre-enable the menu in config.
    $this->config('advanced_mega_menu.settings')
      ->set('enabled_menus', ['test_menu'])
      ->save();

    // Disable the test menu.
    $this->controller->toggle('test_menu', 'disable');

    // Fetch the updated configuration.
    $enabled = $this->config('advanced_mega_menu.settings')
      ->get('enabled_menus');

    // Assert that the menu is now disabled (removed from config).
    $this->assertNotContains('test_menu', $enabled);
  }

}
